/* global __CLIENT__ Hammer */
import {createRef, PureComponent, ReactNode, RefObject} from 'react';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getLayoutOpacityByShift from 'components/MobileSideSheet/utilities/getLayoutOpacityByShift';

import ModalWithHistoryBack from 'containers/withSupportHistoryBack/ModalWithHistoryBack/ModalWithHistoryBack';

import {EModalAnimationType} from 'components/Modal/Modal';

import cx from './MobileSideSheet.scss';

if (__CLIENT__) {
    require('hammerjs');
}

interface IMobileSideSheetProps extends IWithQaAttributes {
    isVisible: boolean;
    onClose: () => void;
    children: ReactNode;
}

interface IMobileSideSheetState extends IWithQaAttributes {
    horizontalShift: number;
    isVisible: boolean;
}

interface IAnimationProps extends IWithQaAttributes {
    opacity: string;
    transform: string;
}

/* Constants */
const SHOW_SHIFT = 0;
const HIDE_SHIFT = 100;
const HIDE_THRESHOLD = 40;

class MobileSideSheet extends PureComponent<
    IMobileSideSheetProps,
    IMobileSideSheetState
> {
    private readonly sideSheetRef: RefObject<HTMLDivElement> =
        createRef<HTMLDivElement>();

    private readonly touchAreaRef: RefObject<HTMLDivElement> =
        createRef<HTMLDivElement>();

    private hammerInstance: HammerManager | null = null;

    private contentWidth: number = 200;

    readonly state: IMobileSideSheetState = {
        horizontalShift: HIDE_SHIFT,
        isVisible: false,
    };

    componentDidUpdate(prevProps: IMobileSideSheetProps): void {
        const {isVisible} = this.props;

        if (isVisible && !prevProps.isVisible) {
            this.setState({isVisible: true}, () => {
                requestAnimationFrame(this.showContent);
            });
        }

        if (!isVisible && prevProps.isVisible) {
            requestAnimationFrame(this.hideContent);
        }
    }

    componentWillUnmount(): void {
        this.unBindEvents();
    }

    /* Actions */
    private initBindings(): void {
        const touchAreaNode = this.touchAreaRef.current;

        if (touchAreaNode) {
            this.hammerInstance = new Hammer(touchAreaNode);
            this.hammerInstance
                .get('pan')
                .set({direction: Hammer.DIRECTION_HORIZONTAL, threshold: 1});
            this.hammerInstance.on(
                'panright panleft',
                this.handlePanHorizontal,
            );
            this.hammerInstance.on('tap', this.handleTap);
            this.hammerInstance.on('panend', this.handlePanEnd);
        }
    }

    private unBindEvents(): void {
        if (this.hammerInstance) {
            this.hammerInstance.off(
                'panright panleft',
                this.handlePanHorizontal,
            );
            this.hammerInstance.off('tap', this.handleTap);
            this.hammerInstance.off('panend', this.handlePanEnd);
            this.hammerInstance.destroy();

            this.hammerInstance = null;
        }
    }

    private onAnimationEnd = (): void => {
        const {onClose} = this.props;
        const {horizontalShift} = this.state;

        if (horizontalShift === HIDE_SHIFT) {
            this.setState({isVisible: false});

            onClose();
        }
    };

    private showContent = (): void => {
        this.initBindings();
        this.setState({horizontalShift: SHOW_SHIFT});
    };

    private hideContent = (): void => {
        this.unBindEvents();
        this.setState({horizontalShift: HIDE_SHIFT});
    };

    private resetHorizontalShift = (): void => {
        this.setState({horizontalShift: SHOW_SHIFT});
    };

    /* Helpers */
    private getCurrentAnimationProps = (): IAnimationProps => {
        const {horizontalShift} = this.state;

        return {
            opacity: getLayoutOpacityByShift(horizontalShift),
            transform: `translateX(${horizontalShift}%)`,
        };
    };

    /* Hammer Events Handlers */
    private handleTap = ({target: targetNode}: TouchInput): void => {
        const sideSheetNode = this.sideSheetRef.current;

        if (sideSheetNode && targetNode) {
            const isSideSheetTap = sideSheetNode.contains(targetNode as Node);

            if (!isSideSheetTap) {
                this.hideContent();
            }
        }
    };

    private handlePanHorizontal = (e: HammerInput): void => {
        const deltaX = Math.max(e.deltaX, 0);
        const horizontalShift = (deltaX / this.contentWidth) * 100;

        this.setState({horizontalShift});
    };

    private handlePanEnd = (): void => {
        const {horizontalShift} = this.state;

        if (horizontalShift > HIDE_THRESHOLD) {
            this.hideContent();
        } else {
            this.resetHorizontalShift();
        }
    };

    render(): ReactNode {
        const {children} = this.props;
        const {isVisible} = this.state;
        const {opacity, transform} = this.getCurrentAnimationProps();

        return (
            <ModalWithHistoryBack
                className={cx('modal')}
                containerClassName={cx('modalContent')}
                isMobile
                fullScreen
                isVisible={isVisible}
                preventBodyScroll={isVisible}
                animation={EModalAnimationType.NONE}
                hasCloseButton={false}
                onClose={this.hideContent}
                {...prepareQaAttributes(this.props)}
            >
                <div className={cx('content')} ref={this.touchAreaRef}>
                    {isVisible && (
                        <div className={cx('layout')} style={{opacity}} />
                    )}
                    <div
                        className={cx('sideSheet')}
                        ref={this.sideSheetRef}
                        style={{transform}}
                        onTransitionEnd={this.onAnimationEnd}
                    >
                        {children}
                    </div>
                </div>
            </ModalWithHistoryBack>
        );
    }
}

export default MobileSideSheet;
