import React, {ReactElement} from 'react';

import {useMobile} from 'utilities/hooks/useMobile';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import ModalWithHistoryBack, {
    IModalWithHistoryBackProps,
} from 'containers/withSupportHistoryBack/ModalWithHistoryBack/ModalWithHistoryBack';

import Modal, {EModalAnimationType} from 'components/Modal/Modal';
import Back from './components/Back/Back';

import cx from './ModalWithBackButton.scss';

interface IModalWithBackButtonProps
    extends Partial<IModalWithHistoryBackProps> {
    children: React.ReactNode;
    isVisible: boolean;
    backButtonText?: string;
    canRenderBackButton: boolean;

    /**
     * В некоторых случаях не нужна поддержка закрытия при браузерном "назад", например,
     * если модалу уже соответствует отдельный урл.
     */
    withHistorySupport?: boolean;
    onClose: () => void;
}

function ModalWithBackButton(props: IModalWithBackButtonProps): ReactElement {
    const {
        children,
        isVisible,
        onClose,
        fullScreen,
        backButtonText,
        canRenderBackButton,
        withHistorySupport = true,
        ...modalProps
    } = props;

    const isMobile = useMobile();

    const canAnimateModal = fullScreen && isMobile;

    const extendedModalProps = {
        isVisible: Boolean(canAnimateModal && isVisible),
        isMobile,
        preventBodyScroll: isVisible,
        animation: canAnimateModal
            ? EModalAnimationType.PUSH_RIGHT
            : EModalAnimationType.APPEAR,
        hasCloseButton: !canRenderBackButton,
        onClose,
        fullScreen,
        ...modalProps,
    };

    const content = (
        <>
            {canRenderBackButton && (
                <Back
                    className={cx('backButton')}
                    onClick={onClose}
                    text={backButtonText}
                    theme="black"
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'backButton',
                    })}
                />
            )}
            {children}
        </>
    );

    if (withHistorySupport) {
        return (
            <ModalWithHistoryBack {...extendedModalProps}>
                {content}
            </ModalWithHistoryBack>
        );
    }

    return <Modal {...extendedModalProps}>{content}</Modal>;
}

export default React.memo(ModalWithBackButton);
