import {memo} from 'react';

import {EProjectName} from 'constants/common';

import {ECommonGoal} from 'utilities/metrika/types/goals/common';
import {IWithClassName} from 'types/withClassName';
import {TMetrikaGoal} from 'utilities/metrika/types/goals/all';

import {reachGoal} from 'utilities/metrika';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useOnNavigateToProject} from 'utilities/hooks/useOnNavigateToProject/useOnNavigateToProject';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';

import Link from 'components/Link/Link';
import {useNavigationLinks} from 'components/NavigationContainer/hooks/useNavigationLinks';
import MoreLinks from 'components/NavigationContainer/components/MoreLinks';

import cx from './NavigationContainer.scss';

export interface INavigationProps extends IWithClassName, IWithQaAttributes {
    collapsed?: boolean;
}

const tabClickMetrikaGoals: PartialRecord<EProjectName, TMetrikaGoal> = {
    [EProjectName.AVIA]: ECommonGoal.AVIA_TAB,
    [EProjectName.TRAINS]: ECommonGoal.TRAINS_TAB,
    [EProjectName.HOTELS]: ECommonGoal.HOTELS_TAB,
    [EProjectName.BUSES]: ECommonGoal.BUS_TAB,
    [EProjectName.TOURS]: ECommonGoal.TOURS_TAB,
};

function NavigationContainer({
    className,
    collapsed,
    ...props
}: INavigationProps) {
    const deviceType = useDeviceType();
    const onNavigateToProject = useOnNavigateToProject();

    const projects = useNavigationLinks(
        useImmutableCallback((project: EProjectName, isActive: boolean) => {
            const goal = isActive
                ? ECommonGoal.ACTIVE_TAB
                : tabClickMetrikaGoals[project];

            if (goal) {
                reachGoal(goal);
            }

            onNavigateToProject(project);
        }),
        collapsed,
    );

    return (
        <nav
            {...prepareQaAttributes(props)}
            className={cx(className, 'root', {
                root_mobile: deviceType.isMobile,
            })}
        >
            {projects.visible.map(
                ({type, link, active, label, onClick, title}) => (
                    <Link
                        key={type}
                        aria-current={active ? 'page' : undefined}
                        className={cx('link', {
                            link_active: active,
                        })}
                        theme="none"
                        to={link}
                        title={title}
                        onClick={onClick}
                        {...prepareQaAttributes({
                            parent: props,
                            current: type,
                        })}
                    >
                        {label}
                    </Link>
                ),
            )}
            {projects.hidden.length > 0 && (
                <MoreLinks links={projects.hidden} />
            )}
        </nav>
    );
}

export default memo(NavigationContainer);
