import {memo, ReactNode, useRef} from 'react';

import {useEventListener} from 'utilities/hooks/useEventListener';

import * as i18n from 'i18n/common';

import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import cx from 'components/NavigationContainer/NavigationContainer.scss';
import ArrowBottomIcon from 'icons/16/ArrowBottom';
import Link from 'components/Link/Link';
import Dropdown from 'components/Dropdown/Dropdown';
import {INavigationLink} from 'components/NavigationContainer/hooks/useNavigationLinks';
import MenuItem from 'components/MenuItem/MenuItem';
import Menu from 'components/Menu/Menu';

export interface IMoreLinksProps {
    links: INavigationLink[];
}

function MoreLinks({links}: IMoreLinksProps) {
    const dropdownRef = useRef<Dropdown>(null);

    useEventListener('scroll', () => dropdownRef.current?.hidePopup());

    return (
        <Dropdown
            ref={dropdownRef}
            renderSwitcher={({
                input: {ref},
                meta: {visible, hidePopup, showPopup},
            }): ReactNode => (
                <TextWithIcon
                    className={cx('link', 'more', 'Link', 'Link_router')}
                    onClick={visible ? hidePopup : showPopup}
                    __dirtyRef={ref}
                    spaceBetween={1}
                    iconRight={ArrowBottomIcon}
                    text={i18n.more({count: links.length})}
                />
            )}
            renderPopup={({meta: {hidePopup}}): ReactNode => (
                <Menu
                    onSelect={hidePopup}
                    initialFocusedName={links.find(link => link.active)?.type}
                >
                    {links.map(
                        ({active, type, link, label, onClick, title}) => (
                            <Link
                                onClick={onClick}
                                to={link}
                                theme="none"
                                title={title}
                                key={type}
                            >
                                <MenuItem
                                    value={type}
                                    enableSelectedIcon
                                    selected={active}
                                >
                                    {label}
                                </MenuItem>
                            </Link>
                        ),
                    )}
                </Menu>
            )}
            popupClassName={cx('popup')}
        />
    );
}

export default memo(MoreLinks);
