import {useLocation} from 'react-router-dom';
import {useMemo} from 'react';

import {EProjectName} from 'constants/common';
import {URLs} from 'constants/urls';
import {
    NAVIGATION_COLLAPSED_MAX_VISIBLE_LINKS,
    NAVIGATION_PROJECTS_ORDER,
} from 'constants/navigation';

import {internalUrl} from 'utilities/url';
import getNavigationProjectText from 'utilities/getNavigationProjectText';

export interface INavigationLink {
    type: string;
    link: string;
    title?: string;
    label: string;
    active: boolean;
    onClick?(): void;
}

export interface IUseNavigationLinksResult {
    visible: INavigationLink[];
    hidden: INavigationLink[];
}

export function useNavigationLinks(
    onProjectClick: (type: EProjectName, active: boolean) => void,
    collapsed?: boolean,
): IUseNavigationLinksResult {
    const {pathname} = useLocation();

    const links = useMemo(
        () => [
            ...NAVIGATION_PROJECTS_ORDER.map(type => ({
                type,
                link: internalUrl(URLs[type]),
                title: undefined,
                label: getNavigationProjectText(type),
                active: pathname.startsWith(URLs[type]),
                onClick: () =>
                    onProjectClick(type, pathname.startsWith(URLs[type])),
            })),
        ],
        [pathname, onProjectClick],
    );

    return useMemo(() => {
        if (!collapsed) {
            return {
                visible: links,
                hidden: [],
            };
        }

        const visible = links.slice(0, NAVIGATION_COLLAPSED_MAX_VISIBLE_LINKS);
        const hidden = links.filter(item => !visible.includes(item));

        return {
            visible,
            hidden,
        };
    }, [links, collapsed]);
}
