import React, {useCallback, useRef} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {URLs} from 'constants/urls';
import {EProjectName} from 'constants/common';
import {NAVIGATION_PROJECTS_ORDER} from 'constants/navigation';

import {IWithClassName} from 'src/types/withClassName';
import {EUserAvatarSize} from 'components/User/types';
import {ETripsGoal} from 'utilities/metrika/types/goals/trips';
import {ECommonGoal} from 'utilities/metrika/types/goals/common';
import {EPubSubEvent} from 'types/EPubSubEvent';

import * as mailActions from 'reducers/common/userUnreadMailCount/actions';

import userFullInfoSelector from 'selectors/common/userFullInfoSelector';
import {isNewTripSelector} from 'selectors/trips/tripPage/isNewTripSelector';

import {params} from 'utilities/metrika';
import {internalUrl} from 'utilities/url';
import getNavigationProjectText from 'utilities/getNavigationProjectText';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import {useOnNavigateToProject} from 'utilities/hooks/useOnNavigateToProject/useOnNavigateToProject';
import {useReachGoal} from 'utilities/metrika/useReachGoal';
import {usePubSubSubscriber} from 'utilities/hooks/usePubSubSubscriber';
import {useWhiteLabelConfig} from 'utilities/hooks/useWhiteLabelConfig';

import * as i18nBlock from 'i18n/account';
import * as i18nNavBlock from 'i18n/common-navigation';

import Flex from 'components/Flex/Flex';
import Link from 'components/Link/Link';
import User from 'components/User/User';
import Separator from 'components/Separator/Separator';
import LinkButton from 'components/LinkButton/LinkButton';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import YandexMessengerLink from 'components/YandexMessenger/components/YandexMessengerLink/YandexMessengerLink';
import MobileSideSheet from 'components/MobileSideSheet/MobileSideSheet';
import NotificationDot from 'components/NotificationDot/NotificationDot';
import Label, {ECornersType, ELabelThemeType} from 'components/Label/Label';
import PlusInfo from './components/PlusInfo/PlusInfo';
import ActiveTripsList from './components/ActiveTripsList/ActiveTripsList';
import BurgerMenuIcon from 'icons/24/Burger';
import GiftOutlineIcon from 'icons/16/GiftOutline';

import useElementClick from './hooks/useElementClick';

import cx from './NavigationSideSheet.scss';

export interface INavigationSideSheetProps
    extends IWithClassName,
        IWithQaAttributes {}

const nonToursProjectsOrder = NAVIGATION_PROJECTS_ORDER.filter(
    project => project !== EProjectName.TOURS,
);

const NavigationSideSheet: React.FC<INavigationSideSheetProps> = props => {
    const {className} = props;

    const dispatch = useDispatch();
    const {userInfo, unreadMailCount} = useSelector(userFullInfoSelector);
    const isNewTrip = useSelector(isNewTripSelector);
    const fetchUserUnreadMailCount = useCallback(() => {
        dispatch(mailActions.fetchUserUnreadMailCount());
    }, [dispatch]);
    const onNavigateToProject = useOnNavigateToProject();
    const rootRef = useRef<HTMLDivElement>(null);

    const {value: isVisible, setTrue: show, setFalse: hide} = useBoolean(false);

    usePubSubSubscriber(EPubSubEvent.OPEN_MOBILE_SIDE_SHEET, show);

    const isAuth = isAuthUser(userInfo);
    const isNewTripDotVisible = isAuth && isNewTrip;

    const handleRootClick: React.MouseEventHandler<HTMLDivElement> =
        useElementClick({
            elementSelector: 'a[href^="/"]',
            rootRef,
            onClick: hide,
        });

    const handleTripsLinkClick = useReachGoal(
        ETripsGoal.TRIPS_LINK_NAVIGATION_SIDE_SHEET_CLICK,
    );

    const handleCertificateClick = useCallback(() => {
        params({common: {sideSheetCertificateClick: true}});
    }, []);

    const {isWhiteLabel} = useWhiteLabelConfig();

    return (
        <>
            <LinkButton
                className={className}
                onClick={show}
                theme="black"
                {...prepareQaAttributes({
                    parent: props,
                    current: 'toggleButton',
                })}
            >
                <NotificationDot size="m" dotVisible={isNewTripDotVisible}>
                    <BurgerMenuIcon />
                </NotificationDot>
            </LinkButton>
            <MobileSideSheet isVisible={isVisible} onClose={hide}>
                <div
                    className={cx('navigationSideSheet')}
                    ref={rootRef}
                    onClick={handleRootClick}
                    {...prepareQaAttributes(props)}
                >
                    <User
                        className={cx('user')}
                        userInfo={userInfo}
                        avatarSize={EUserAvatarSize.M}
                        unreadMailCount={unreadMailCount}
                        withName
                        fetchUserUnreadMailCount={fetchUserUnreadMailCount}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'user',
                        })}
                    />
                    {!isWhiteLabel && (
                        <Flex tag="section" flexDirection="column" between={6}>
                            {nonToursProjectsOrder.map(
                                (projectName): React.ReactNode => (
                                    <Link
                                        className={cx('link')}
                                        theme="black"
                                        key={projectName}
                                        to={internalUrl(URLs[projectName])}
                                        onClick={(): void =>
                                            onNavigateToProject(projectName)
                                        }
                                        {...prepareQaAttributes({
                                            parent: props,
                                            current: 'projectLink',
                                            key: projectName,
                                        })}
                                    >
                                        {getNavigationProjectText(projectName)}
                                    </Link>
                                ),
                            )}
                            <Link
                                className={cx('link', 'newLink')}
                                theme="black"
                                to={internalUrl(URLs[EProjectName.TOURS])}
                                rel="noopener noreferrer"
                            >
                                {getNavigationProjectText(EProjectName.TOURS)}
                                <Label
                                    theme={ELabelThemeType.ATTENTION}
                                    cornersType={ECornersType.ROUNDED}
                                >
                                    {i18nNavBlock.newItem()}
                                </Label>
                            </Link>
                            <Link
                                className={cx('link', 'newLink')}
                                theme="black"
                                url={internalUrl(URLs.promoCertificatesUrl)}
                                target="_blank"
                                rel="noopener noreferrer"
                                onClick={handleCertificateClick}
                            >
                                <TextWithIcon
                                    text={i18nNavBlock.certificates()}
                                    iconLeft={GiftOutlineIcon}
                                />
                                <Label
                                    theme={ELabelThemeType.ATTENTION}
                                    cornersType={ECornersType.ROUNDED}
                                >
                                    {i18nNavBlock.newItem()}
                                </Label>
                            </Link>
                        </Flex>
                    )}

                    {!isWhiteLabel && <Separator margin={7} />}

                    <Flex tag="section" flexDirection="column" between={7}>
                        <Flex flexDirection="column" between={4}>
                            <Link
                                to={internalUrl(URLs.accountTrips)}
                                className={cx('link')}
                                theme="black"
                                onClick={handleTripsLinkClick}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'tripsLink',
                                })}
                            >
                                <Flex alignItems="center">
                                    <span className={cx('tripLinkText')}>
                                        {i18nBlock.menuDotTrips()}
                                    </span>

                                    {isNewTripDotVisible && (
                                        <div className={cx('newTripDot')} />
                                    )}
                                </Flex>
                            </Link>
                            {isAuth && (
                                <ActiveTripsList
                                    {...prepareQaAttributes({
                                        parent: props,
                                        current: 'activeTripsList',
                                    })}
                                />
                            )}
                        </Flex>

                        {isAuth && (
                            <Link
                                to={internalUrl(URLs.accountPassengers)}
                                className={cx('link')}
                                theme="black"
                            >
                                {i18nBlock.menuDotPassengers()}
                            </Link>
                        )}
                    </Flex>
                    <Separator margin={7} />
                    {!isWhiteLabel && <PlusInfo userInfo={userInfo} />}

                    <YandexMessengerLink
                        className={cx('messenger')}
                        label={i18nBlock.chatWithSupport()}
                        labelSize="m"
                        entrypoint="sideSheet"
                        metrikaGoal={ECommonGoal.PORTAL_CHAT_MENU_CLICK}
                    />
                </div>
            </MobileSideSheet>
        </>
    );
};

export default React.memo(NavigationSideSheet);
