import React, {useEffect} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import {ETripsGoal} from 'utilities/metrika/types/goals/trips';

import {fetchActiveTrips} from 'reducers/trips/activeTrips/thunk';

import activeTripsSelector from 'selectors/trips/activeTrips/activeTripsSelector';

import {getTripListItemKey} from 'projects/trips/utilities/getTripListItemKey';
import {useReachGoal} from 'utilities/metrika/useReachGoal';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Flex from 'components/Flex/Flex';
import ActiveTripsListItem from './components/ActiveTripsListItem/ActiveTripsListItem';

interface IActiveTripsListProps extends IWithClassName, IWithQaAttributes {}

const ActiveTripsList: React.FC<IActiveTripsListProps> = props => {
    const {className} = props;

    const dispatch = useDispatch();
    const trips = useSelector(activeTripsSelector);

    useEffect(() => {
        dispatch(fetchActiveTrips());
    }, [dispatch]);

    const handleTripLinkClick = useReachGoal(
        ETripsGoal.TRIP_LINK_NAVIGATION_SIDE_SHEET_CLICK,
    );

    if (!trips.length) {
        return null;
    }

    return (
        <Flex
            className={className}
            flexDirection="column"
            between={2}
            {...prepareQaAttributes(props)}
        >
            {trips.map(trip => {
                const key = getTripListItemKey(trip);

                return (
                    <ActiveTripsListItem
                        key={key}
                        trip={trip}
                        onClick={handleTripLinkClick}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'activeTripsListItem',
                            key,
                        })}
                    />
                );
            })}
        </Flex>
    );
};

export default ActiveTripsList;
