import {useCallback} from 'react';

import {unknownToErrorOrUndefined} from 'utilities/error';
import {logWarning} from 'utilities/logger/logWarning';

interface IParams {
    elementSelector: string;
    rootRef: React.RefObject<HTMLElement>;
    onClick(): void;
}

/**
 * Hook, который возвращает обработчик, который отлавливает события на элементах
 * подходящих под селектор elementSelector или их детей, которые находятся внутри rootRef
 * и вызывает onClick
 * @link https://learn.javascript.ru/event-delegation
 */
export default function useElementClick({
    elementSelector,
    rootRef,
    onClick,
}: IParams): React.MouseEventHandler<HTMLDivElement> {
    return useCallback(
        e => {
            try {
                const root = rootRef.current;

                if (!root) {
                    return;
                }

                if (!(e.target instanceof HTMLElement)) {
                    return;
                }

                const target = e.target;
                const element = target.closest(elementSelector);

                if (!element || !root.contains(element)) {
                    return;
                }

                onClick();
            } catch (err) {
                logWarning(
                    {
                        message:
                            '[YATRAVEL][useElementClick] Error when use useElementClick',
                    },
                    unknownToErrorOrUndefined(err),
                );
            }
        },
        [elementSelector, rootRef, onClick],
    );
}
