import React, {useCallback, useMemo, useRef} from 'react';

import {IWithClassName} from 'types/withClassName';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceModMobile} from 'utilities/stylesUtils';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {usePrint} from 'utilities/hooks/usePrint';

import * as i18nBlock from 'i18n/happyPage';

import Box from 'components/Box/Box';
import PrintIcon from 'icons/16/Print';
import Button from 'components/Button/Button';
import DownloadIcon from 'icons/16/Download';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import CancelIcon from 'icons/16/Cancel';
import KebabIcon from 'icons/16/KebabVertical';
import MessageBoxPopup from 'components/MessageBoxPopup/MessageBoxPopup';

import cx from './OrderActions.scss';

export enum EOrderActionsSource {
    ORDER_PAGE = 'ORDER_PAGE',
    HAPPY_PAGE = 'HAPPY_PAGE',
}

export interface IOrderActionsDetails {
    url: string;
    onClick?(): void;
}

type TRenderIconFunction = (
    className?: string,
    defaultIcon?: JSX.Element,
) => JSX.Element | undefined;

export interface IOrderActionsPrint {
    title?: string;
    url: string;
    renderIcon?: TRenderIconFunction;
    isDisabled?: boolean;
    onClick?(): void;
}

export interface IOrderActionsDownload {
    title?: string;
    url: string;
    fileName?: string;
    renderIcon?: TRenderIconFunction;
    isDisabled?: boolean;
    onClick?(): void;
}

export interface IOrderActionsCancel {
    title: string;
    isDisabled?: boolean;
    tooltipTitle?: string;
    renderIcon?: TRenderIconFunction;
    onClick(): void;
}

interface IOrderActionsProps extends IWithClassName, IWithQaAttributes {
    source: EOrderActionsSource;
    orderDetails?: IOrderActionsDetails;
    print?: IOrderActionsPrint;
    download?: IOrderActionsDownload;
    cancel?: IOrderActionsCancel;
    more?: {
        isDisabled?: boolean;
        innerRef: React.RefObject<HTMLElement>;
        onClick(): void;
    };
}

const printIcon = <PrintIcon />;
const downloadIcon = <DownloadIcon className={cx('icon')} />;
const cancelIcon = <CancelIcon className={cx('icon')} />;
const moreIcon = <KebabIcon />;

const TOOLTIP_DIRECTIONS = [EPopupDirection.BOTTOM];

const OrderActions: React.FC<IOrderActionsProps> = props => {
    const {className, source, orderDetails, print, download, cancel, more} =
        props;

    const deviceType = useDeviceType();
    const {isPrinting, startPrinting} = usePrint({
        printable: print?.url,
        type: 'pdf',
    });
    const {
        value: isCancelTooltipVisible,
        setTrue: openCancelTooltip,
        setFalse: closeCancelTooltip,
    } = useBoolean(false);

    const rootQa = getQa(props);

    const cancelTooltipRef = useRef<HTMLDivElement>(null);

    const handlePrintClick = useCallback((): void => {
        if (!print) {
            return;
        }

        startPrinting();
        print.onClick?.();
    }, [print, startPrinting]);

    const orderDetailsButton = useMemo(() => {
        if (!orderDetails) {
            return null;
        }

        return (
            <ButtonLink
                to={orderDetails.url}
                theme="secondary"
                size={deviceType.isMobile ? 'l' : 'm-inset'}
                width={deviceType.isMobile ? 'max' : 'auto'}
                onClick={orderDetails.onClick}
                {...prepareQaAttributes({
                    parent: rootQa,
                    current: 'detailsLink',
                })}
            >
                {deviceType.isMobile
                    ? i18nBlock.orderDetailsButton2()
                    : i18nBlock.orderDetailsButton()}
            </ButtonLink>
        );
    }, [orderDetails, deviceType, rootQa]);

    const printButton = useMemo(() => {
        if (!print || deviceType.isMobile) {
            return null;
        }

        return (
            <Button
                {...prepareQaAttributes({
                    parent: rootQa,
                    current: 'printButton',
                })}
                theme="secondary"
                size="m-inset"
                onClick={handlePrintClick}
                disabled={print.isDisabled || isPrinting}
                width={deviceType.isMobile ? 'max' : 'auto'}
                icon={
                    print.renderIcon
                        ? print.renderIcon(cx('icon'), printIcon)
                        : printIcon
                }
            >
                {print.title || i18nBlock.print()}
            </Button>
        );
    }, [print, isPrinting, handlePrintClick, deviceType, rootQa]);

    const downloadButton = useMemo(() => {
        if (!download) {
            return null;
        }

        return (
            <ButtonLink
                theme={
                    source === EOrderActionsSource.ORDER_PAGE
                        ? 'primary'
                        : 'secondary'
                }
                size={deviceType.isMobile ? 'l' : 'm-inset'}
                width={deviceType.isMobile ? 'max' : 'auto'}
                url={download.url}
                download={download.fileName}
                icon={
                    download.renderIcon
                        ? download.renderIcon(cx('icon'), downloadIcon)
                        : downloadIcon
                }
                disabled={download.isDisabled}
                onClick={download.onClick}
                {...prepareQaAttributes({
                    parent: rootQa,
                    current: 'downloadButton',
                })}
            >
                {download.title || i18nBlock.download()}
            </ButtonLink>
        );
    }, [download, source, deviceType.isMobile, rootQa]);

    const cancelButton = useMemo(() => {
        if (!cancel) {
            return null;
        }

        return (
            <div
                className={cx('cancelTooltipWrapper')}
                onClick={cancel.isDisabled ? openCancelTooltip : undefined}
                ref={cancelTooltipRef}
            >
                <Button
                    theme="secondary"
                    size={deviceType.isMobile ? 'l' : 'm-inset'}
                    width={deviceType.isMobile ? 'max' : 'auto'}
                    onClick={cancel.onClick}
                    disabled={cancel.isDisabled}
                    icon={
                        cancel.renderIcon
                            ? cancel.renderIcon(cx('icon'), cancelIcon)
                            : cancelIcon
                    }
                    {...prepareQaAttributes({
                        parent: rootQa,
                        current: 'cancelButton',
                    })}
                >
                    {cancel.title}
                </Button>

                {cancel.tooltipTitle && (
                    <MessageBoxPopup
                        isVisible={isCancelTooltipVisible}
                        anchorRef={cancelTooltipRef}
                        direction={TOOLTIP_DIRECTIONS}
                        onClose={closeCancelTooltip}
                    >
                        <Box x={3} y={2}>
                            {cancel.tooltipTitle}
                        </Box>
                    </MessageBoxPopup>
                )}
            </div>
        );
    }, [
        cancel,
        openCancelTooltip,
        deviceType.isMobile,
        rootQa,
        isCancelTooltipVisible,
        closeCancelTooltip,
    ]);

    const moreButton = useMemo(() => {
        if (!more) {
            return null;
        }

        return (
            <Button
                theme="secondary"
                disabled={more.isDisabled}
                size={deviceType.isMobile ? 'l' : 'm-inset'}
                onClick={more.onClick}
                icon={moreIcon}
                innerRef={more.innerRef}
            />
        );
    }, [more, deviceType]);

    const visibleButtonsCount =
        (orderDetailsButton ? 1 : 0) +
        (cancelButton ? 1 : 0) +
        (printButton ? 1 : 0) +
        (downloadButton ? 1 : 0) +
        (moreButton ? 0.5 : 0);

    return (
        <Box
            className={cx(
                'root',
                deviceModMobile('root', deviceType),
                {root_showIconsOnAnyMobileScreen: visibleButtonsCount <= 1.5},
                className,
            )}
            inline
            nowrap
            between="2"
        >
            {orderDetailsButton}
            {cancelButton}
            {printButton}
            {downloadButton}
            {moreButton}
        </Box>
    );
};

export default React.memo(OrderActions);
