import React from 'react';

import {EProjectName} from 'constants/common';

import {IWithClassName} from 'types/withClassName';
import EDisplayOrderState from 'types/common/orders/EDisplayOrderState';

import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceModMobile} from 'utilities/stylesUtils';
import getOrderStatusByDisplayState from 'components/OrderHeader/utilities/getOrderStatusByDisplayState';

import * as i18nBlock from 'i18n/account-OrderHeader';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import OrderIcon from 'projects/account/components/OrderIcon/OrderIcon';
import OrderStatus from 'projects/account/components/OrderStatus/OrderStatus';
import OrderBackLink from 'projects/account/pages/Order/components/OrderBackLink/OrderBackLink';

import cx from './OrderHeader.scss';

interface IOrderHeader extends IWithClassName, IWithQaAttributes {
    number?: string;
    type:
        | EProjectName.TRAINS
        | EProjectName.HOTELS
        | EProjectName.AVIA
        | EProjectName.BUSES;
    displayState?: EDisplayOrderState;
    tripId?: string | null;
}

const OrderHeader: React.FC<IOrderHeader> = props => {
    const {className, number, type, displayState, tripId} = props;

    const deviceType = useDeviceType();

    return (
        <Flex
            className={cx(
                'root',
                deviceModMobile('root', deviceType),
                className,
            )}
            flexDirection={deviceType.isMobile ? 'column' : 'row'}
            alignItems={deviceType.isMobile ? 'flex-start' : 'center'}
            justifyContent="space-between"
            {...prepareQaAttributes(props)}
        >
            <Flex className={cx('backPart')} alignItems="center">
                <OrderBackLink tripId={tripId} />
            </Flex>
            {Boolean(number) && (
                <Flex
                    className={cx('numberPart')}
                    alignItems="center"
                    inline
                    between={2}
                >
                    <div
                        className={cx('numberBlock')}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'numberBlock',
                        })}
                    >
                        <OrderIcon type={type} className={cx('icon')} />

                        <Text size="m" color="primary">
                            {i18nBlock._number({
                                number: number ?? '',
                            })}
                        </Text>
                    </div>

                    {displayState && (
                        <OrderStatus
                            status={getOrderStatusByDisplayState(displayState)}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'status',
                            })}
                        />
                    )}
                </Flex>
            )}
        </Flex>
    );
};

export default React.memo(OrderHeader);
