import {FunctionComponent, useCallback} from 'react';

import {IWithClassName} from 'types/withClassName';

import * as i18nBlock from 'i18n/hotelsExtranet';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import PageInput from './components/PageInput';
import ArrowLeftIcon from 'icons/16/ArrowLeft';
import ArrowRightIcon from 'icons/16/ArrowRight';

import cx from './PaginationInput.scss';

interface IPaginationInputProps extends IWithClassName {
    pageNumber: number;
    minValue: number;
    maxValue: number;
    onPageChange?(page: number): void;
    onPageInput?(): void;
    onArrowClick?(): void;
}

const PaginationInput: FunctionComponent<IPaginationInputProps> = ({
    className,
    pageNumber,
    minValue,
    maxValue,
    onPageChange,
    onPageInput,
    onArrowClick,
}) => {
    const setPage = useCallback(
        newPage => {
            if (pageNumber !== newPage) {
                onPageChange?.(newPage);
            }
        },
        [pageNumber, onPageChange],
    );

    const handlePageInput = useCallback(
        newPage => {
            setPage(newPage);
            onPageInput?.();
        },
        [setPage, onPageInput],
    );

    const handleLeftClick = useCallback(() => {
        if (pageNumber > minValue) {
            setPage(pageNumber - 1);
        }

        onArrowClick?.();
    }, [setPage, minValue, pageNumber, onArrowClick]);

    const handleRightClick = useCallback(() => {
        if (pageNumber < maxValue) {
            setPage(pageNumber + 1);
        }

        onArrowClick?.();
    }, [setPage, pageNumber, maxValue, onArrowClick]);

    return (
        <Flex
            className={className}
            flexDirection="row"
            alignItems="center"
            between={3}
            inline
        >
            <PageInput
                className={cx('pageInput')}
                page={pageNumber}
                minValue={minValue}
                maxValue={maxValue}
                onChange={handlePageInput}
            />
            <Text>
                {i18nBlock._of()} {maxValue}
            </Text>
            <ArrowLeftIcon
                className={cx(
                    'arrow',
                    pageNumber <= minValue && 'arrow_disabled',
                )}
                onClick={handleLeftClick}
            />
            <ArrowRightIcon
                className={cx(
                    'arrow',
                    pageNumber >= maxValue && 'arrow_disabled',
                )}
                onClick={handleRightClick}
            />
        </Flex>
    );
};

export default PaginationInput;
