import {FC, useCallback, useEffect, useState} from 'react';
import {debounce} from 'lodash';

import {IWithClassName} from 'types/withClassName';

import Input from 'components/Input/Input';

interface IPageInputProps extends IWithClassName {
    page: number;
    minValue: number;
    maxValue: number;
    onChange: (page: number) => void;
}

const PageInput: FC<IPageInputProps> = ({
    className,
    page,
    minValue,
    maxValue,
    onChange,
}) => {
    const [pageInternal, setPageInternal] = useState<number | undefined>(page);

    useEffect(() => {
        setPageInternal(page);
    }, [page]);

    const debouncedChange = useCallback(
        debounce<(newPage: number | undefined) => void>(newPage => {
            if (newPage) {
                onChange(newPage);
            }
        }, 2000),
        [onChange],
    );

    const handlePageInputChange = useCallback(
        (e, pageStr) => {
            let newPage = parseInt(pageStr, 10);

            if (isNaN(newPage)) {
                setPageInternal(undefined);
                debouncedChange(undefined);

                return;
            }

            if (newPage < minValue) {
                newPage = minValue;
            } else if (maxValue && newPage > maxValue) {
                newPage = maxValue;
            }

            setPageInternal(newPage);
            debouncedChange(newPage);
        },
        [setPageInternal, debouncedChange, minValue, maxValue],
    );

    return (
        <Input
            className={className}
            value={pageInternal || ''}
            onChange={handlePageInputChange}
        />
    );
};

export default PageInput;
