import React, {useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';

import IPrice from 'utilities/currency/PriceInterface';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import isEqualPrice from 'utilities/price/isEqualPrice';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';

import * as i18nBlock from 'i18n/account-OrderHotels-Price';

import Price from 'components/Price/Price';

interface IPartialPriceProps extends IWithClassName, IWithQaAttributes {
    /**
     * Часть цены, например сколько оплачено
     */
    price: IPrice | undefined;
    /**
     * Стоимость заказа
     */
    totalPrice: IPrice;
}

const PartialPrice: React.FC<IPartialPriceProps> = props => {
    const {className, price, totalPrice} = props;

    const rootQa = getQa(props);

    const priceNode = useMemo((): React.ReactNode => {
        if (!price || isEqualPrice(price, totalPrice)) {
            return (
                <Price
                    {...totalPrice}
                    {...prepareQaAttributes({
                        parent: rootQa,
                        current: 'totalPrice',
                    })}
                />
            );
        }

        return insertJSXIntoKey(i18nBlock.amountPaid)({
            paidPriceNode: (
                <Price
                    {...price}
                    {...prepareQaAttributes({
                        parent: rootQa,
                        current: 'price',
                    })}
                />
            ),
            totalPriceNode: (
                <Price
                    {...totalPrice}
                    {...prepareQaAttributes({
                        parent: rootQa,
                        current: 'totalPrice',
                    })}
                />
            ),
        });
    }, [rootQa, price, totalPrice]);

    return (
        <span className={className} {...prepareQaAttributes(rootQa)}>
            {priceNode}
        </span>
    );
};

export default PartialPrice;
