import {FC, ReactNode, memo, useCallback} from 'react';

import {IWithClassName} from 'types/withClassName';
import IPartnerRequisites from 'components/PartnersRequisites/types/IPartnerRequisites';

import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/trains-partners-requisites';

import ModalWithHistoryBack from 'containers/withSupportHistoryBack/ModalWithHistoryBack/ModalWithHistoryBack';

import LinkButton from 'components/LinkButton/LinkButton';
import Modal from 'components/Modal/Modal';
import PartnersInfo from 'components/PartnersRequisites/components/PartnersInfo/PartnersInfo';
import {ILinkProps} from '@yandex-lego/components/Link';

interface IPartnersRequisitesProps extends IWithClassName, IWithQaAttributes {
    partners: IPartnerRequisites[];
    renderTextWithLink?: (linkProps: ILinkProps) => ReactNode;
    modalTitle?: string;
    modalSubTitle?: string;
}

const PartnersRequisites: FC<IPartnersRequisitesProps> = props => {
    const {
        className,
        partners,
        modalTitle = i18nBlock.title(),
        modalSubTitle,
    } = props;

    const deviceType = useDeviceType();

    const defaultRenderTextWithLink = useCallback((linkProps: ILinkProps) => {
        return insertJSXIntoKey(i18nBlock.dataDashDescription)({
            link: (
                <LinkButton {...linkProps}>{i18nBlock.partners()}</LinkButton>
            ),
        });
    }, []);

    const renderTextWithLink =
        props.renderTextWithLink ?? defaultRenderTextWithLink;

    const {
        setTrue: handleOpen,
        setFalse: handleClose,
        value: isVisible,
    } = useBoolean(false);

    return (
        <div className={className} {...prepareQaAttributes(props)}>
            {renderTextWithLink({
                onClick: handleOpen,
                ...prepareQaAttributes({
                    parent: props,
                    current: 'openLink',
                }),
            })}

            <ModalWithHistoryBack
                isMobile={deviceType.isMobile}
                isVisible={isVisible}
                onClose={handleClose}
                {...prepareQaAttributes({parent: props, current: 'modal'})}
            >
                <Modal.Content>
                    <PartnersInfo
                        partners={partners}
                        modalTitle={modalTitle}
                        modalSubTitle={modalSubTitle}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'partnersInfo',
                        })}
                    />
                </Modal.Content>
            </ModalWithHistoryBack>
        </div>
    );
};

export default memo(PartnersRequisites);
