import React, {useCallback, useMemo, useState} from 'react';

import {YANDEX_PLUS_URL} from 'constants/urls/plus';
import {EAppActions} from 'constants/platforms/TPlatforms';

import {IWithClassName} from 'types/withClassName';

import {deviceModMobile} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {CHAR_MINUS} from 'utilities/strings/charCodes';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlockAviaPlus from 'i18n/avia-plusPromo2021';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import YandexPlusSignColorGradientS from 'icons/illustrations/YandexPlusSignColorGradient/YandexPlusSignColorGradientS';
import YandexPlusSignColorGradientM from 'icons/illustrations/YandexPlusSignColorGradient/YandexPlusSignColorGradientM';
import TriggerDetailsButton from 'components/TriggerDetailsButton/TriggerDetailsButton';
import Heading from 'components/Heading/Heading';
import {EMessageBoxPopupTheme} from 'components/MessageBoxPopup/MessageBoxPopup';
import Link from 'components/Link/Link';
import {TBoxSizes} from 'components/Box/Box';

import ScopeContext from 'contexts/ScopeContext';
import {useCoordinator} from 'contexts/PlatformContext';

import cx from './PlusInfo.scss';

interface IPlusInfoProps extends IWithClassName, IWithQaAttributes {
    plusPoints: number;
    popupText: string;
    popupTitle: string;
    isDiscount?: boolean;
    size?: 's' | 'm';
    direction?: React.CSSProperties['flexDirection'];
    justifyContent?: React.CSSProperties['justifyContent'];
    between?: TBoxSizes;
}

const PlusInfo: React.FC<IPlusInfoProps> = props => {
    const {
        plusPoints,
        className,
        popupText,
        popupTitle,
        isDiscount,
        size = 's',
        direction,
        justifyContent = 'space-between',
        between,
    } = props;
    const deviceType = useDeviceType();
    const [ref, setRef] = useState(undefined);

    const iconSize = useMemo(() => {
        switch (size) {
            case 's':
                return 12;
            case 'm':
            default:
                return 16;
        }
    }, [size]);

    const coordinator = useCoordinator();
    const handlePlusDetailsLinkClick: React.MouseEventHandler = useCallback(
        e => {
            const processed = coordinator.doAction(
                EAppActions.NAVIGATE_TO_EXTERNAL_URL,
                YANDEX_PLUS_URL,
            );

            if (processed) {
                e.preventDefault();
            }
        },
        [coordinator],
    );

    const TriggerDetailsButtonAsComponent = useMemo(() => {
        const Component: React.FC = () => {
            return (
                <TriggerDetailsButton
                    theme={EMessageBoxPopupTheme.WHITE}
                    iconClassName={cx(
                        'detailsButtonIcon',
                        `detailsButtonIcon_${size}`,
                        isDiscount && 'detailsButtonIcon_hidden',
                    )}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'triggerDetailsButton',
                    })}
                >
                    <div
                        className={cx(
                            'popupContainer',
                            deviceModMobile('popupContainer', deviceType),
                        )}
                    >
                        <Heading level="3">{popupTitle}</Heading>
                        <Text tag="div" className={cx('popupContainer_text')}>
                            {popupText}
                        </Text>
                        <Link
                            className={cx('popupContainer_link')}
                            url={YANDEX_PLUS_URL}
                            onClick={handlePlusDetailsLinkClick}
                        >
                            {i18nBlockAviaPlus.plusPopupMoreInfo()}
                        </Link>
                    </div>
                </TriggerDetailsButton>
            );
        };

        return Component;
    }, [
        deviceType,
        handlePlusDetailsLinkClick,
        isDiscount,
        popupText,
        popupTitle,
        props,
        size,
    ]);

    return (
        <ScopeContext.Provider value={ref || undefined}>
            <Flex
                className={cx('plusInfoContainer', className)}
                flexDirection={direction}
                justifyContent={justifyContent}
                between={between}
                innerRef={setRef}
                {...prepareQaAttributes(props)}
            >
                <TextWithIcon
                    size={size}
                    spaceBetween={1}
                    text={
                        isDiscount
                            ? i18nBlockAviaPlus.createOrderWithdrawText()
                            : i18nBlockAviaPlus.createOrderRefundText()
                    }
                    iconRight={TriggerDetailsButtonAsComponent}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'label',
                    })}
                />
                {isDiscount ? (
                    <Text
                        color="plus"
                        size={size}
                        weight="medium"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'discountPrice',
                        })}
                    >
                        {plusPoints ? `${CHAR_MINUS} ${plusPoints}` : 0}
                    </Text>
                ) : (
                    <TextWithIcon
                        text={plusPoints || 0}
                        size={size}
                        spaceBetween="1"
                        iconLeftProps={{
                            className: cx('plusIcon', `plusIcon_${size}`),
                        }}
                        iconLeft={
                            size === 'm'
                                ? YandexPlusSignColorGradientM
                                : YandexPlusSignColorGradientS
                        }
                        iconSize={iconSize}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'plusPoints',
                        })}
                    />
                )}
            </Flex>
        </ScopeContext.Provider>
    );
};

export default PlusInfo;
