import React, {useCallback, useEffect, useRef} from 'react';
import {useInView} from 'react-intersection-observer';

import {YANDEX_PLUS_URL} from 'constants/urls/plus';
import {EAppActions} from 'constants/platforms/TPlatforms';

import {IWithClassName} from 'types/withClassName';
import EPopupDirection from 'components/Popup/types/EPopupDirection';
import {TBadgeType} from 'types/hotels/offer/IHotelOffer';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {
    getIconBySizeAndId,
    getIconSize,
    getIconSizes,
} from 'components/PlusInfoPopup/utilities/sizeUtils';
import getLabel from 'components/PlusInfoPopup/utilities/getLabel';

import * as i18nBlock from 'i18n/components-PlusInfoPopup';

import {EMessageBoxPopupTheme} from 'components/MessageBoxPopup/MessageBoxPopup';
import Link from 'components/Link/Link';
import LinkButton from 'components/LinkButton/LinkButton';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import MessageBoxPopupOrBottomSheet from 'components/MessageBoxPopupOrBottomSheet/MessageBoxPopupOrBottomSheet';

import {useCoordinator} from 'contexts/PlatformContext';

import cx from './PlusInfoPopup.scss';

const POPUP_POSITIONS = [EPopupDirection.TOP, EPopupDirection.BOTTOM];
const POPUP_ANIMATION_DURATION = 1000;

export type TSize = 's-inset' | 'm';

export type TLabelDescriptionType = 'short' | 'full';

interface IPlusInfoPopupProps extends IWithClassName, IWithQaAttributes {
    iconClassName?: string;
    /** Количество баллов Плюса */
    points?: number;
    eligible?: boolean;
    popupPositions?: EPopupDirection[];

    labelSize?: TSize;
    labelDescriptionType?: TLabelDescriptionType;

    title?: string;
    description?: string;
    linkUrl?: string;
    linkText?: string;
    buttonText?: string;
    onPlusLinkClick?: (e: React.MouseEvent<Element, MouseEvent>) => void;
    badgeId?: TBadgeType;
}

const PlusInfoPopup: React.FC<IPlusInfoPopupProps> = props => {
    const {onPlusLinkClick, badgeId: id} = props;

    const coordinator = useCoordinator();
    const handlePlusLinkClick: React.MouseEventHandler = useCallback(
        e => {
            const processed = coordinator.doAction(
                EAppActions.NAVIGATE_TO_EXTERNAL_URL,
                YANDEX_PLUS_URL,
            );

            if (processed) {
                e.preventDefault();
            }

            if (onPlusLinkClick) {
                onPlusLinkClick(e);
            }
        },
        [coordinator, onPlusLinkClick],
    );

    const {title, description, linkUrl, linkText} = props;

    const defaultContent = (
        <>
            <h3 className={cx('header')}>{title || i18nBlock.header()}</h3>
            <div className={cx('description')}>
                <pre className={cx('descriptionPre')}>
                    {description ||
                        (props.eligible
                            ? i18nBlock.descriptionPlus()
                            : i18nBlock.descriptionV2())}
                </pre>
            </div>
            <Link
                className={cx('link')}
                theme="normal"
                url={linkUrl || YANDEX_PLUS_URL}
                onClick={handlePlusLinkClick}
            >
                {linkText || i18nBlock.moreInfoLink()}
            </Link>
        </>
    );
    const {
        className,
        iconClassName,
        children = defaultContent,
        points: plusPoints,
        popupPositions,
        labelSize = 's-inset',
        labelDescriptionType = 'short',
        buttonText,
    } = props;

    const rootRef = useRef(null);
    const {value: isVisible, setTrue: show, setFalse: hide} = useBoolean(false);

    const {
        value: canCheckVisibility,
        setTrue: withCheckVisibility,
        setFalse: withoutCheckVisibility,
    } = useBoolean(false);
    const {ref: contentRef, inView: isContentInView} = useInView({
        rootMargin: '-70px 0px 0px 0px',
        threshold: 0.9,
    });

    useEffect((): void => {
        if (isVisible) {
            setTimeout(withCheckVisibility, POPUP_ANIMATION_DURATION);
        } else {
            setTimeout(withoutCheckVisibility, POPUP_ANIMATION_DURATION);
        }
    }, [isVisible, withCheckVisibility, withoutCheckVisibility]);

    useEffect((): void => {
        if (canCheckVisibility && !isContentInView) {
            hide();
        }
    }, [canCheckVisibility, isContentInView, hide]);

    const handleClickShow = useCallback(
        e => {
            e.stopPropagation();
            show();
        },
        [show],
    );

    const handleClickHide = useCallback(
        (
            e:
                | React.MouseEvent<HTMLButtonElement, MouseEvent>
                | Event
                | undefined,
        ) => {
            e?.stopPropagation?.();

            hide();
        },
        [hide],
    );

    const boundaryRef =
        typeof document === 'undefined' ? undefined : {current: document.body};

    return (
        <>
            <LinkButton
                theme="black"
                innerRef={rootRef}
                className={cx('root', className)}
                onClick={handleClickShow}
                {...prepareQaAttributes(props)}
            >
                <TextWithIcon
                    textClassName={cx('plusLabel')}
                    iconLeft={getIconBySizeAndId(labelSize, id)}
                    iconSize={getIconSize(labelSize)}
                    iconLeftClassName={iconClassName}
                    iconLeftProps={getIconSizes(labelSize, id)}
                    text={
                        buttonText || getLabel(labelDescriptionType, plusPoints)
                    }
                    size={labelSize}
                    spaceBetween={1}
                />
            </LinkButton>
            <MessageBoxPopupOrBottomSheet
                isVisible={isVisible}
                onClose={handleClickHide}
                messageBoxProps={{
                    theme: EMessageBoxPopupTheme.WHITE,
                    direction: popupPositions || POPUP_POSITIONS,
                    anchorRef: rootRef,
                    boundaryRef,
                }}
            >
                <div className={cx('content')} ref={contentRef}>
                    {children}
                </div>
            </MessageBoxPopupOrBottomSheet>
        </>
    );
};

export default PlusInfoPopup;
