import React, {useCallback} from 'react';

import {URLs} from 'constants/urls';
import {EAppActions} from 'constants/platforms/TPlatforms';

import {IWithClassName} from 'types/withClassName';

import {reloadPage} from 'utilities/browser/reloadPage';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/components-PortalServerError';

import Button from 'components/Button/Button';
import ButtonLink from 'components/ButtonLink/ButtonLink';

import {useCoordinator, usePlatform} from 'contexts/PlatformContext';

import cx from './PortalServerError.scss';

interface IPortalServerErrorProps extends IWithClassName {}

const PortalServerError: React.FC<IPortalServerErrorProps> = ({className}) => {
    const {isMobile} = useDeviceType();
    const {isTravelApp} = usePlatform();
    const coordinator = useCoordinator();

    const handleClickReloadButton = useCallback(() => {
        reloadPage();
    }, []);

    const handleActionButtonClick = useCallback<
        React.MouseEventHandler<HTMLElement>
    >(
        event => {
            if (isTravelApp) {
                const processed = coordinator.doAction(
                    EAppActions.NAVIGATE_TO_INDEX_PAGE,
                );

                if (processed) {
                    event.preventDefault();
                }
            }
        },
        [coordinator, isTravelApp],
    );

    return (
        <section
            className={cx('portalServerError', className, {
                portalServerError_mobile: isMobile,
            })}
        >
            <div className={cx('content')}>
                <h2 className={cx('title')}>{i18nBlock.title()}</h2>
                <div className={cx('description')}>
                    <div
                        className={cx('descriptionText')}
                        dangerouslySetInnerHTML={{
                            __html: i18nBlock.description(),
                        }}
                    />
                </div>
                <div className={cx('buttons')}>
                    <Button
                        className={cx('button', 'reloadButton')}
                        onClick={handleClickReloadButton}
                        size={isMobile ? 'l' : 'm'}
                    >
                        {i18nBlock.reloadButton()}
                    </Button>

                    <ButtonLink
                        className={cx('button')}
                        theme="primary"
                        size={isMobile ? 'l' : 'm'}
                        url={URLs.index}
                        onClick={handleActionButtonClick}
                    >
                        {i18nBlock.toIndexPage()}
                    </ButtonLink>
                </div>
            </div>
        </section>
    );
};

export default PortalServerError;
