import {FC, useMemo} from 'react';

import IPrice from 'types/common/price/IPrice';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/common';

import ErrorModal, {
    IErrorModalProps,
    TErrorActionType,
} from 'components/ErrorModal/ErrorModal';
import BookPriceInfo from 'components/BookPriceInfo/BookPriceInfo';

import cx from './PriceErrorModal.scss';

interface IPriceErrorModalProps
    extends Omit<IErrorModalProps, 'primaryAction' | 'secondaryAction'> {
    price?: IPrice;
    lastPrice?: IPrice;
    isDisabled?: boolean;
    retryButtonText?: string;
    cancelButtonText?: string;
    onRetryClick: () => void;
    onCancelClick: () => void;
    retryUrl?: string;
}

const PriceErrorModal: FC<IPriceErrorModalProps> = ({
    price,
    lastPrice,
    isDisabled,
    retryButtonText,
    cancelButtonText,
    onRetryClick,
    onCancelClick,
    retryUrl,
    ...errorModalProps
}) => {
    const primaryAction = useMemo((): TErrorActionType => {
        const title = retryButtonText || i18nBlock.pay();
        const props = {
            className: cx('button'),
            disabled: isDisabled,
        };

        return retryUrl
            ? {
                  type: 'button-link',
                  title,
                  handler: onRetryClick,
                  props: {
                      ...props,
                      url: retryUrl,
                  },
              }
            : {
                  type: 'button',
                  title,
                  handler: onRetryClick,
                  props,
              };
    }, [onRetryClick, retryButtonText, isDisabled, retryUrl]);

    const secondaryAction = useMemo(
        (): TErrorActionType => ({
            type: 'button',
            title: cancelButtonText || i18nBlock.cancelOrder(),
            handler: onCancelClick,
            props: {
                className: cx('button'),
                disabled: isDisabled,
            },
        }),
        [onCancelClick, cancelButtonText, isDisabled],
    );

    return (
        <ErrorModal
            {...errorModalProps}
            primaryAction={primaryAction}
            secondaryAction={secondaryAction}
            {...prepareQaAttributes(errorModalProps)}
        >
            {price && Boolean(price) && (
                <BookPriceInfo
                    className={cx('priceInfo')}
                    text={lastPrice ? undefined : i18nBlock.payTitle()}
                    price={price}
                    lastPrice={lastPrice}
                    {...prepareQaAttributes(errorModalProps)}
                />
            )}
        </ErrorModal>
    );
};

export default PriceErrorModal;
