import React, {
    ChangeEvent,
    ChangeEventHandler,
    Children,
    ReactElement,
    ReactNode,
} from 'react';

import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import {
    RadioButton as RadioButtonBase,
    IRadioButtonOption,
} from '@yandex-lego/components/RadioButton/desktop';

import cx from './RadioButton.scss';

/* Props */
export enum ERadioButtonState {
    NORMAL = 'normal',
    ERROR = 'error',
}

export type TOnChangeRadioButton = ChangeEventHandler<HTMLInputElement>;

export interface IRadioButtonProps extends IWithClassName, IWithQaAttributes {
    onChange: TOnChangeRadioButton;
    children: ReactNode;
    tabIndex?: number;
    id?: string;
    radioRef?: (ref: HTMLInputElement | null) => void;
    value?: string | number;
    size?: 's' | 'm' | 'm-inset' | 'l' | 'xl';
    name?: string;
    disabled?: boolean;
    state?: ERadioButtonState;
    onBlur?: React.FocusEventHandler<HTMLElement>;
    onFocus?: React.FocusEventHandler<HTMLElement>;
    width?: 'auto' | 'max';
    theme?: 'default' | 'between';
}

interface IRadioButtonState {
    selectedValue: string;
    focused: boolean;
}

interface IRadioButtonRadio
    extends Omit<IRadioButtonOption, 'value'>,
        IWithClassName {
    children: ReactNode;
    onMouseEnter?: (event: React.MouseEvent, value?: unknown) => void;
    onMouseLeave?: (event: React.MouseEvent, value?: unknown) => void;
    onKeyDown?: React.KeyboardEventHandler;
    onBlur?: React.FocusEventHandler;
    onFocus?: React.FocusEventHandler;
    mainValue?: string;
    mainDisabled?: boolean;
    checked?: boolean;
    controlAttrs?: React.HTMLAttributes<HTMLDivElement> & IWithQaAttributes;
    value: string | number;
}

const RadioButtonRadio: React.FC<IRadioButtonRadio> = ({
    className,
    value,
    children,
    disabled,
    mainValue,
    mainDisabled,
    checked,
    controlAttrs,
    ...rest
}) => {
    return (
        <div
            className={cx('radio', className, {
                radio_checked: checked || mainValue === value.toString(),
                radio_disabled: mainDisabled ? true : disabled,
            })}
            {...controlAttrs}
            {...rest}
        >
            {children}
        </div>
    );
};

class RadioButton extends React.PureComponent<
    IRadioButtonProps,
    IRadioButtonState
> {
    static Radio = RadioButtonRadio;

    static ERROR = ERadioButtonState.ERROR;
    static NORMAL = ERadioButtonState.NORMAL;

    readonly state: IRadioButtonState = {
        selectedValue: this.props.value?.toString() || '',
        focused: false,
    };

    componentDidUpdate(prevProps: Readonly<IRadioButtonProps>): void {
        if (prevProps.value !== this.props.value) {
            const selectedValue = this.props.value?.toString() || '';

            this.setState({selectedValue});
        }
    }

    private handleFocusElement = (
        event: React.FocusEvent<HTMLInputElement>,
    ): void => {
        const {onFocus} = this.props;

        if (typeof onFocus === 'function') {
            onFocus(event);
        }

        this.setState({focused: true});
    };

    private handleBlurElement = (
        event: React.FocusEvent<HTMLElement>,
    ): void => {
        const {onBlur} = this.props;

        if (typeof onBlur === 'function') {
            onBlur(event);
        }

        this.setState({
            focused: false,
        });
    };

    private onChange = (event: ChangeEvent<HTMLInputElement>): void => {
        this.setState({selectedValue: event.target.value});

        const {onChange} = this.props;

        if (onChange) {
            onChange(event);
        }
    };

    render(): ReactNode {
        const {
            name,
            className,
            size = 'm',
            state = ERadioButtonState.NORMAL,
            disabled = false,
            children,
            width = 'auto',
            theme = 'default',
        } = this.props;
        const {selectedValue} = this.state;

        return (
            <div
                onFocus={this.handleFocusElement}
                onBlur={this.handleBlurElement}
                className={cx(className)}
                {...prepareQaAttributes(this.props)}
            >
                <RadioButtonBase
                    // @ts-ignore
                    size={size}
                    onChange={this.onChange}
                    view="default"
                    name={name}
                    value={selectedValue}
                    disabled={disabled}
                    options={Children.map(
                        children as ReactElement,
                        (item: ReactElement) => {
                            const qaProp = prepareQaAttributes({
                                parent: item.props.controlAttrs,
                                current: 'text',
                            });

                            return {
                                value: item.props.value.toString(),
                                disabled: item.props.disabled,
                                controlRef: item.props.controlRef,
                                autoFocus: false,
                                children: React.cloneElement(item, {
                                    mainDisabled: disabled,
                                    mainValue: selectedValue,
                                    autoFocus: false,
                                    ...qaProp,
                                }),
                                ...item.props.controlAttrs,
                            };
                        },
                    )}
                    className={cx(
                        'container',
                        `container_size_${size}`,
                        width === 'max' && 'container_width_max',
                        `container_theme_${theme}`,
                        {
                            container_error: state === ERadioButtonState.ERROR,
                            container_focused: this.state.focused,
                        },
                    )}
                />
            </div>
        );
    }
}

export default RadioButton;
