import React, {Children, ReactElement, ReactNode} from 'react';
import memoize from 'lodash/memoize';

import {IWithClassName} from 'types/withClassName';

import RadioButtonSlideOption, {
    IRadioButtonSlideOptionProps,
} from './components/RadioButtonSlideOption';

import cx from './RadioButtonSlide.scss';

export interface IRadioButtonSlideProps extends IWithClassName {
    children:
        | ReactElement<IRadioButtonSlideOptionProps>
        | ReactElement<IRadioButtonSlideOptionProps>[];
    value?: string;
    onChange?(value: string): void;
    disabled?: boolean;
    isLoading?: boolean;
    size?: 'm' | 'l';
    name?: string;
    backgroundClassName?: string;
}

interface IRadioButtonSlideState {
    value: string;
    currentIndex: number;
}

class RadioButtonSlide extends React.PureComponent<
    IRadioButtonSlideProps,
    IRadioButtonSlideState
> {
    state = {
        value: '',
        currentIndex: 0,
    };

    private getTabStyle = memoize(
        (itemsLength: number, currentIndex: number) => {
            const leftSideShift = currentIndex === 0 ? 1 : 0;
            const rightSideShift = itemsLength - 1 === currentIndex ? 1 : 0;

            return {
                width: `calc(${100 / itemsLength}% - ${rightSideShift}px)`,
                left: `calc(${
                    (100 / itemsLength) * currentIndex
                }% + ${leftSideShift}px)`,
            };
        },
        (itemsLength: number, currentIndex: number) =>
            `${itemsLength}.${currentIndex}`,
    );

    componentDidMount(): void {
        const {value} = this.props;

        if (value) {
            this.setNewValue(value);
        }
    }

    componentDidUpdate(prevProps: Readonly<IRadioButtonSlideProps>): void {
        const {value: prevValue} = prevProps;
        const {value} = this.props;
        const {value: stateValue} = this.state;

        if (value && value !== prevValue && value !== stateValue) {
            this.setNewValue(value);
        }
    }

    static Option = RadioButtonSlideOption;

    private setNewValue(value: string): void {
        const {children} = this.props;
        let newIndex = 0;

        Children.forEach(
            children as ReactElement<IRadioButtonSlideOptionProps>,
            (item, index) => {
                if (item.props.value === value) {
                    newIndex = index;
                }
            },
        );
        this.setState({
            value,
            currentIndex: newIndex,
        });
    }

    private handleItemClick =
        (
            index: number,
            itemCallback: IRadioButtonSlideOptionProps['onClick'],
        ) =>
        (value: string): void => {
            const {onChange} = this.props;

            this.setState({
                value,
                currentIndex: index,
            });
            onChange?.(value);
            itemCallback?.(value);
        };

    render(): ReactNode {
        const {
            className,
            children,
            size = 'l',
            disabled,
            backgroundClassName,
        } = this.props;
        const {value, currentIndex} = this.state;
        const itemsLength = Children.count(children);

        return (
            <span
                className={cx('root', className, {
                    size_m: size === 'm',
                    size_l: size === 'l',
                    disabled,
                })}
                role="radio-group"
            >
                {Children.map(
                    children as ReactElement<IRadioButtonSlideOptionProps>,
                    (item, index) =>
                        React.cloneElement(item, {
                            ...item.props,
                            key: item.props.value,
                            className: cx('item', item.props.className, {
                                checked: item.props.value === value,
                                disabled: item.props.disabled || disabled,
                            }),
                            checked: item.props.value === value,
                            disabled: item.props.disabled || disabled,
                            onClick: this.handleItemClick(
                                index,
                                item.props.onClick,
                            ),
                            width: item.props.width || `${100 / itemsLength}%`,
                        }),
                )}
                <div
                    className={cx('tab')}
                    style={this.getTabStyle(itemsLength, currentIndex)}
                />
                <div className={cx('background', backgroundClassName)} />
            </span>
        );
    }
}

export default RadioButtonSlide;
