import React, {PureComponent} from 'react';
import {SpringValues, Transition, animated} from 'react-spring';

import {IWithClassName} from 'types/withClassName';

import cx from './SearchControl.scss';

const ANIMATE_TRANSITION_BASE_CONFIG = {
    initial: null,
    from: {opacity: 0, display: 'none'},
    enter: {opacity: 1, display: 'block'},
    leave: {opacity: 0, display: 'none'},
    unique: true,
    reset: true,
};

export enum ESearchControlPlaceholderPosition {
    TOP = 'top',
    BOTTOM = 'bottom',
    CENTER = 'center',
}

export interface ISearchControlProps extends IWithClassName {
    placeholder: string;
    controlNode: React.ReactNode;
    isEmpty: boolean;
    placeholderClassName: string;
    placeholderPosition?: ESearchControlPlaceholderPosition;
}

export interface ISearchControlState {
    canUseTransition: boolean;
}

/**
 * Делает анимацию перехода от незаполненного контрола с плейсхолдером
 * в заполненный контрол + плейсхолдер над значением.
 **/
class SearchControl extends PureComponent<
    ISearchControlProps,
    ISearchControlState
> {
    static readonly defaultProps: Partial<ISearchControlProps> = {
        className: '',
        placeholderClassName: '',
        controlNode: null,
        isEmpty: true,
        placeholderPosition: ESearchControlPlaceholderPosition.TOP,
    };

    readonly state: ISearchControlState = {canUseTransition: false};

    componentDidMount(): void {
        this.setState({canUseTransition: true});
    }

    private renderLabelPlaceholder(): React.ReactNode {
        const {placeholder, isEmpty, placeholderPosition} = this.props;

        if (
            placeholderPosition === ESearchControlPlaceholderPosition.CENTER &&
            !isEmpty
        )
            return null;

        return (
            <div
                className={cx(
                    'labelPlaceholder',
                    {
                        labelPlaceholder_visible: !isEmpty,
                    },
                    `labelPlaceholder_${placeholderPosition}`,
                )}
            >
                {placeholder}
            </div>
        );
    }

    private renderControlPlaceholder = (
        placeholder: string,
        styles: SpringValues = {},
    ): React.ReactElement => {
        const {placeholderClassName} = this.props;

        return (
            <animated.div
                className={cx('controlPlaceholder', placeholderClassName)}
                style={styles}
            >
                {placeholder}
            </animated.div>
        );
    };

    private renderControl(): React.ReactNode {
        const {canUseTransition} = this.state;
        const {placeholder, controlNode, isEmpty, placeholderPosition} =
            this.props;

        return (
            <div
                className={cx(
                    'control',
                    {
                        control_position_center:
                            isEmpty ||
                            placeholderPosition ===
                                ESearchControlPlaceholderPosition.CENTER,
                    },
                    `control_position_${placeholderPosition}`,
                )}
            >
                {controlNode}
                {isEmpty &&
                    (canUseTransition ? (
                        <Transition
                            items={isEmpty}
                            {...ANIMATE_TRANSITION_BASE_CONFIG}
                        >
                            {(styles): React.ReactElement =>
                                this.renderControlPlaceholder(
                                    placeholder,
                                    styles,
                                )
                            }
                        </Transition>
                    ) : (
                        this.renderControlPlaceholder(placeholder)
                    ))}
            </div>
        );
    }

    render(): React.ReactNode {
        const {className} = this.props;

        return (
            <div className={cx('controlContainer', className)}>
                {this.renderLabelPlaceholder()}
                {this.renderControl()}
            </div>
        );
    }
}

export default SearchControl;
