import React, {useCallback, useEffect, useMemo, useRef} from 'react';
import moment from 'moment';

import {
    ICalendarPricesByType,
    TCalendarPrices,
} from 'types/common/calendarPrice/ICalendarPrice';
import {
    ESearchFormFieldName,
    ESearchFormSize,
    ESearchFormTriggerViewType,
} from 'components/SearchForm/types';
import {IWithClassName} from 'types/withClassName';
import {ECalendarType} from 'components/Calendar/types';
import {WHEN_SPECIAL_VALUE} from 'types/common/When';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import getUTCDateByDate from 'components/Calendar/utilities/getUTCDateByDate';
import addYears from 'components/Calendar/utilities/addYears';
import getPreparedDateForTrigger from 'components/Calendar/utilities/getPreparedDateForTrigger';
import {ROBOT} from 'utilities/dateUtils/formats';
import {deviceModMobile} from 'utilities/stylesUtils';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import getFormattedDate from 'components/SearchForm/components/DatePicker/utilities/getFormattedDate';
import isSpecialWhen from 'utilities/dateUtils/when/isSpecialWhen';
import getWhenSpecialValueText from 'utilities/dateUtils/when/getWhenSpecialValueText';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import CommonDatePicker, {
    DatePicker as CommonDatePickerClass,
} from 'components/DatePicker/DatePicker';
import CommonNewDatePickerClass from 'components/DatePicker/DatePickerNew';
import DatepickerToolbar from 'components/SearchForm/components/DatePicker/components/DatepickerToolbar/DatepickerToolbar';
import {useNewCalendarEnabled} from 'components/DatePicker/hooks/useNewCalendarEnabled';
import {TDatePickerTriggerValueFn} from 'components/DatePicker/components/DatePickerTrigger/DatePickerTriggerNew';
import {
    IDatePickerFooterProps,
    IDatePickerFooterPropsByCalendarType,
} from 'components/DatePicker/components/DatePickerFooter/DatePickerFooter';
import {ESearchControlPlaceholderPosition} from 'components/SearchControl/SearchControl';

import cx from './DatePicker.scss';

interface IDatePickerProps extends IWithClassName, IWithQaAttributes {
    triggerClassName: string;
    triggerFocusClassName: string;
    triggerViewType: ESearchFormTriggerViewType;
    size: ESearchFormSize;
    isExpanded?: boolean;
    headerBlock?: React.ReactNode;
    canAutoHideCalendar?: boolean;
    footerBlockParams?: IDatePickerFooterProps;
    footerBlockParamsByType?: IDatePickerFooterPropsByCalendarType;
    startDate: string | null;
    startDateError?: string[] | boolean;
    error: string[] | boolean;
    startDatePlaceholder: string;
    setStartDate(startDate: string | null): void;
    startDateTrailingValue?: string;
    endDate?: string | null;
    endDateError?: string[] | boolean;
    endDatePlaceholder?: string;
    setEndDate?(startDate: string | null): void;
    hasEndDate: boolean;
    calendarPrices: TCalendarPrices | undefined;
    calendarPricesByType?: ICalendarPricesByType;
    setErrorFieldRefByName: any;
    initiallyCalendarIsOpen?: boolean;
    specialWhenButtons?: WHEN_SPECIAL_VALUE[];
    setFieldRefByName: any;
    isRoundTrip: boolean;
    storeFieldIfNotFilled(typeAndValue: {
        type: ESearchFormFieldName.START_DATE;
        value: string | null;
    }): void;
    requestCalendar?(type?: ECalendarType): void;
    handleRestoreValue(type: ESearchFormFieldName): void;
    onInteract?(val: boolean): void;
    onDateClick?(args?: {
        startDate: Date | null;
        endDate?: Date | null;
        calendarType?: ECalendarType;
    }): void;
    onResetValue?(calendarType?: ECalendarType): void;
    onFastDateClick?: (value: string) => void;
    hideDateClearButton: boolean;
    canToggleDropdowns: boolean;
    onShowCalendar?(calendarType: ECalendarType): void;
    onHideCalendar?(): void;
    isFullDateFormat?: boolean;
    resetRangeAfterFirstInteraction?: boolean;
    datePickerPlaceholderPosition?: ESearchControlPlaceholderPosition;
}

const MIN_DATE = getUTCDateByDate(new Date());
const MAX_DATE = addYears(MIN_DATE, 1);
const NOW_DATE = new Date();

const DatePicker: React.FC<IDatePickerProps> = props => {
    const {
        className,
        triggerClassName,
        triggerFocusClassName,
        size,
        triggerViewType,
        headerBlock: headerBlockFromProps,
        footerBlockParams,
        footerBlockParamsByType,
        canAutoHideCalendar,
        startDate,
        startDateError,
        startDatePlaceholder,
        setStartDate,
        startDateTrailingValue,
        datePickerPlaceholderPosition,
        endDate,
        endDateError,
        endDatePlaceholder,
        setEndDate,
        hasEndDate,
        isRoundTrip,
        isExpanded,
        setErrorFieldRefByName,
        setFieldRefByName,
        storeFieldIfNotFilled,
        initiallyCalendarIsOpen,
        calendarPrices,
        calendarPricesByType,
        requestCalendar,
        handleRestoreValue,
        specialWhenButtons,
        hideDateClearButton,
        canToggleDropdowns,
        isFullDateFormat,
        onDateClick,
        onInteract,
        onShowCalendar,
        onHideCalendar,
        error,
        onFastDateClick,
        resetRangeAfterFirstInteraction,
        onResetValue,
    } = props;

    const deviceType = useDeviceType();

    const isNewCalendarEnabled = useNewCalendarEnabled();

    const datePickerInstanceRef = useRef<
        (CommonDatePickerClass | CommonNewDatePickerClass) | null
    >(null);

    const setDatePickerInstance = useImmutableCallback(
        (node: CommonDatePickerClass | CommonNewDatePickerClass) => {
            datePickerInstanceRef.current = node;

            if (!node) {
                return;
            }

            const calendarRefs = [setFieldRefByName, setErrorFieldRefByName];

            let calendarRefNames;

            if (isNewCalendarEnabled) {
                const dateTriggerRef = (node as CommonNewDatePickerClass)
                    .dateTriggerRef;

                calendarRefNames = {
                    [ESearchFormFieldName.START_DATE]: dateTriggerRef,
                    [ESearchFormFieldName.END_DATE]: dateTriggerRef,
                };
            } else {
                const {startDateTriggerRef, endDateTriggerRef} =
                    node as CommonDatePickerClass;

                calendarRefNames = {
                    [ESearchFormFieldName.START_DATE]: startDateTriggerRef,
                    [ESearchFormFieldName.END_DATE]: endDateTriggerRef,
                };
            }

            Object.entries(calendarRefNames).forEach(
                ([refName, triggerRef]) => {
                    calendarRefs.forEach(calendarRef => {
                        const buttonNode =
                            triggerRef?.current?.buttonRef?.current;

                        if (buttonNode) {
                            calendarRef(refName)(buttonNode);
                        }
                    });
                },
            );
        },
    );

    const storeStartDate = useImmutableCallback(() => {
        storeFieldIfNotFilled({
            type: ESearchFormFieldName.START_DATE,
            value: startDate,
        });
    });

    const handleFastLinkClick = useImmutableCallback((value: string) => {
        storeStartDate();

        setStartDate(value);

        onDateClick?.();

        onFastDateClick?.(value);

        datePickerInstanceRef.current?.hidePopup();
    });

    const handleCalendarInteraction = useImmutableCallback(
        (type: ECalendarType) => {
            requestCalendar?.(type);
            onShowCalendar?.(type);
        },
    );

    const handleResetValue = useImmutableCallback(
        (calendarType: ECalendarType) => {
            handleCalendarInteraction(calendarType);
            onResetValue?.(calendarType);
        },
    );

    const triggerValueOld = useMemo(() => {
        if (hasEndDate) {
            return undefined;
        }

        if (startDate === null) {
            return '';
        }

        if (isSpecialWhen(startDate)) {
            return getWhenSpecialValueText(startDate);
        }

        return getPreparedDateForTrigger(new Date(startDate), {
            isShortFormatDate: false,
        });
    }, [hasEndDate, startDate]);

    const triggerValue: TDatePickerTriggerValueFn = useCallback(
        ({hoveredDate}) => {
            if (hasEndDate || !startDate || hoveredDate) {
                return undefined;
            }

            const startString = startDate.toString();

            if (startString === null) {
                return '';
            }

            if (isSpecialWhen(startString)) {
                return getWhenSpecialValueText(startString);
            }

            return getPreparedDateForTrigger(new Date(startString), {
                isShortFormatDate: false,
            });
        },
        [hasEndDate, startDate],
    );

    const handleDaySelect = useImmutableCallback(
        ({
            startDate: selectedStartDate,
            endDate: selectedEndDate,
            calendarType,
        }: {
            startDate: Date | null;
            endDate?: Date | null;
            calendarType: ECalendarType;
        }) => {
            storeStartDate();

            setStartDate(
                selectedStartDate
                    ? moment.utc(selectedStartDate).format(ROBOT)
                    : null,
            );

            setEndDate?.(
                selectedEndDate
                    ? moment.utc(selectedEndDate).format(ROBOT)
                    : null,
            );

            // Очень странный хак, потому что в текущей реализации формы
            // при выборе даты в календаре не происходит onBlur с кнопки дэйтпикера.
            onInteract?.(false);

            onDateClick?.({
                startDate: selectedStartDate,
                endDate: selectedEndDate,
                calendarType: isNewCalendarEnabled ? calendarType : undefined,
            });

            if (isNewCalendarEnabled) {
                requestCalendar?.();
            }
        },
    );

    const handleBlurDatePicker = useImmutableCallback(() => {
        onInteract?.(false);
    });

    const handleFocusDatePicker = useImmutableCallback(() => {
        onInteract?.(true);
    });

    const formattedStartDate = useMemo(() => {
        return getFormattedDate(startDate);
    }, [startDate]);

    const formattedEndDate = useMemo(() => {
        if (!endDate) {
            return null;
        }

        return getFormattedDate(endDate);
    }, [endDate]);

    useEffect(() => {
        if (initiallyCalendarIsOpen) {
            datePickerInstanceRef.current?.showPopup(ECalendarType.StartDate);
        }
    }, [initiallyCalendarIsOpen]);

    const headerBlock = useMemo(() => {
        if (headerBlockFromProps) {
            return headerBlockFromProps;
        }

        if (!specialWhenButtons) {
            return null;
        }

        return (
            <DatepickerToolbar
                onClick={handleFastLinkClick}
                specialWhenButtons={specialWhenButtons}
            />
        );
    }, [handleFastLinkClick, headerBlockFromProps, specialWhenButtons]);

    return (
        <CommonDatePicker
            className={cx(
                'root',
                deviceModMobile('root', deviceType),
                className,
            )}
            size={size}
            triggerViewType={triggerViewType}
            ref={setDatePickerInstance}
            triggerClassName={triggerClassName}
            triggerFocusClassName={triggerFocusClassName}
            headerNode={headerBlock}
            footerBlockParams={footerBlockParams}
            footerBlockParamsByType={footerBlockParamsByType}
            canAutoHideCalendar={canAutoHideCalendar}
            datePickerPlaceholderPosition={datePickerPlaceholderPosition}
            startDate={formattedStartDate}
            startDatePlaceholder={startDatePlaceholder}
            startDateTrailingValue={startDateTrailingValue}
            errorStartDate={startDateError}
            endDate={formattedEndDate}
            endDatePlaceholder={endDatePlaceholder}
            errorEndDate={endDateError}
            error={error}
            hasEndDate={hasEndDate}
            isRoundTrip={isRoundTrip}
            maxDate={MAX_DATE}
            minDate={MIN_DATE}
            nowDate={NOW_DATE}
            tabIndex={isExpanded ? 0 : -1}
            triggerValue={isNewCalendarEnabled ? triggerValue : triggerValueOld}
            prices={calendarPrices}
            pricesByType={calendarPricesByType}
            isModalView={deviceType.isMobile}
            deviceType={deviceType}
            canRenderResetIcon={deviceType.isDesktop && !hideDateClearButton}
            isShortTriggerFormatDate={deviceType.isDesktop && !isFullDateFormat}
            canToggle={canToggleDropdowns}
            onShowCalendar={handleCalendarInteraction}
            onHideCalendar={onHideCalendar}
            onResetValue={handleResetValue}
            onDaySelect={handleDaySelect}
            onFocus={handleFocusDatePicker}
            onBlur={handleBlurDatePicker}
            onModalHistoryBack={handleRestoreValue as any}
            resetRangeAfterFirstInteraction={resetRangeAfterFirstInteraction}
            {...prepareQaAttributes(props)}
        />
    );
};

export default React.memo(DatePicker);
