import React, {ReactNode, useCallback, useMemo} from 'react';

import {
    ESearchFormFieldName,
    TSearchFormTypeAndValue,
} from 'components/SearchForm/types';
import typedReactMemo from 'types/typedReactMemo';
import {IWithClassName} from 'types/withClassName';
import {
    ISelectItemOptions,
    ISelectItemPreparedOptions,
} from 'components/SearchForm/types/ISelectItemOptions';
import {TOnShowSuggests} from 'components/SearchSuggest/types/TOnShowSuggests';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import SearchFormFieldModal from 'components/SearchFormFieldModal/SearchFormFieldModal';
import BaseSearchSuggest, {
    ESuggestSource,
    IBaseSuggestItem,
    ISuggestValue,
    TSelectedValue,
} from 'components/SearchSuggest/SearchSuggest';
import {ISearchFormFieldsRefsAndErrors} from 'components/SearchForm/hooks/useFieldsRefsAndErrors';

import cx from './Suggest.scss';

export type TNameSuggest = ESearchFormFieldName.FROM | ESearchFormFieldName.TO;

interface ISuggestProps<SuggestItem extends IBaseSuggestItem>
    extends IWithClassName,
        IWithQaAttributes {
    triggerClassName: string;
    triggerFocusClassName: string;
    searchFormName: string;
    name: TNameSuggest;
    nextFieldName: ESearchFormFieldName.TO | ESearchFormFieldName.START_DATE;
    value: ISuggestValue<SuggestItem>;
    suggestItems: SuggestItem[];
    previousSuggestItems: SuggestItem[] | undefined;
    uniqueValueName: keyof SuggestItem;
    isExpanded?: boolean;
    error: string;
    placeholder: string;
    defaultAutoSelectIndex?: number;
    isSuggestLoading?: boolean;
    getSuggestTitleAndDescription(item: SuggestItem): {
        title: string;
        description: string;
    };
    renderSuggestItemTitle?: (item: SuggestItem) => ReactNode;
    renderSuggestItemDescription?: (item: SuggestItem) => ReactNode;
    renderSuggestItemIcon?: (item: SuggestItem) => ReactNode;
    otherField: ISuggestValue<SuggestItem>['selectedValue'];
    setFieldRefByName: ISearchFormFieldsRefsAndErrors['setFieldRefByName'];
    setErrorFieldRefByName: ISearchFormFieldsRefsAndErrors['setErrorFieldRefByName'];
    setPreviousSearchFormValue?: any;
    focusNextFieldByName: ISearchFormFieldsRefsAndErrors['focusNextFieldByName'];
    setPoint(point: ISuggestValue<SuggestItem>): void;
    storeField(typeAndValue: TSearchFormTypeAndValue): void;
    restoreField(type: ESearchFormFieldName): void;
    onInteract?(val: boolean): void;
    onInputFocus?(
        name: ESearchFormFieldName.FROM | ESearchFormFieldName.TO,
    ): void;
    onInputBlur?(
        name: ESearchFormFieldName.FROM | ESearchFormFieldName.TO,
    ): void;
    handleModalButtonClick?({
        buttonType,
        fieldType,
    }: {
        buttonType: string;
        fieldType: ESearchFormFieldName;
    }): void;
    onSelectItem?(
        item: TSelectedValue<SuggestItem>,
        options?: ISelectItemPreparedOptions,
    ): void;
    canFocusNextField?: (type: string, item: SuggestItem | false) => boolean;
    sortItems?: (a: SuggestItem, b: SuggestItem) => number;
    onShowSuggests?: TOnShowSuggests<TNameSuggest, SuggestItem>;
    validateOnMount?: boolean;
}

const Suggest = <SuggestItem extends IBaseSuggestItem>(
    props: ISuggestProps<SuggestItem>,
): React.ReactElement => {
    const {
        className,
        triggerClassName,
        triggerFocusClassName,
        searchFormName,
        name,
        nextFieldName,
        value: {inputValue, selectedValue},
        suggestItems,
        previousSuggestItems,
        uniqueValueName,
        isExpanded,
        error,
        placeholder,
        getSuggestTitleAndDescription,
        renderSuggestItemTitle,
        renderSuggestItemDescription,
        renderSuggestItemIcon,
        otherField,
        setFieldRefByName,
        setErrorFieldRefByName,
        focusNextFieldByName,
        setPoint,
        storeField,
        restoreField,
        handleModalButtonClick,
        setPreviousSearchFormValue: setPreviousSearchFormValueFromProps,
        onInteract,
        onInputFocus,
        onInputBlur,
        onSelectItem,
        canFocusNextField,
        defaultAutoSelectIndex = 0,
        isSuggestLoading,
        sortItems,
        onShowSuggests,
        validateOnMount,
        ...rest
    } = props;

    const deviceType = useDeviceType();

    const handleFocusSuggest = useImmutableCallback(() => {
        onInputFocus?.(name);
        onInteract?.(true);
    });

    const handleBlurSuggest = useImmutableCallback(() => {
        onInputBlur?.(name);
        onInteract?.(false);
    });

    const handleSuggestFinishSelect = useCallback(
        (type?: string, item?: SuggestItem | false) => {
            if (
                !type ||
                !item ||
                !canFocusNextField ||
                canFocusNextField(type, item)
            ) {
                setTimeout(() => focusNextFieldByName(nextFieldName));
            }
        },
        [focusNextFieldByName, canFocusNextField, nextFieldName],
    );

    const handleChangeSuggestInputValue = useImmutableCallback(
        (suggestValue: ISuggestValue<SuggestItem>) => {
            setPoint(suggestValue);
        },
    );

    const handleSelectSuggestValue = useImmutableCallback(
        (suggestValue: ISuggestValue<SuggestItem>) => {
            storeField({type: name, value: suggestValue});

            setPoint(suggestValue);
        },
    );

    const setPreviousSearchFormValue = useImmutableCallback(() => {
        if (!selectedValue) {
            return;
        }

        setPreviousSearchFormValueFromProps?.({
            fieldType: name,
            fieldValue: selectedValue,
            uniqueValueName,
            formName: searchFormName,
        });
    });

    const handleHideSuggestDropdown = useImmutableCallback(() => {
        setPreviousSearchFormValue();
    });

    const handleResetValue = useImmutableCallback(() => {
        if (inputValue) {
            handleChangeSuggestInputValue({
                inputValue: '',
                source: ESuggestSource.INPUT,
                selectedValue: false,
            });
        }
    });

    const resetInputValue = useCallback(() => {
        setTimeout(() => {
            if (!suggestItems?.length) {
                handleResetValue();
            }
        }, 10);
    }, [suggestItems, handleResetValue]);

    const handleOnModalHistoryBack = useCallback(
        (fieldType: string) => {
            if (!inputValue) {
                restoreField(fieldType as ESearchFormFieldName);
            }

            resetInputValue();
        },
        [inputValue, resetInputValue, restoreField],
    );

    const handleSelectItem = useImmutableCallback(
        async (value: SuggestItem | false, options?: ISelectItemOptions) => {
            if (!value) {
                return;
            }

            const {title} = getSuggestTitleAndDescription(value);
            const preparedOptions: ISelectItemPreparedOptions = {
                isUserInput: true,
                isManualClick: true,
                isTrustedUser: false,
                ...options,
            };
            const {isUserInput, isManualClick, isTrustedUser} = preparedOptions;

            handleSelectSuggestValue({
                selectedValue: value,
                inputValue: title,
                source: ESuggestSource.SUGGESTS,
            });

            await onSelectItem?.(value, {
                isUserInput,
                isManualClick,
                isTrustedUser,
            });
        },
    );

    const handleChangeInputValue = useImmutableCallback((value: string) => {
        handleChangeSuggestInputValue({
            inputValue: value,
            source: ESuggestSource.INPUT,
            selectedValue: false,
        });
    });

    const handleCancelClick = useCallback(
        (fieldType: string) => {
            if (!inputValue) {
                restoreField(fieldType as ESearchFormFieldName);
            }
        },
        [restoreField, inputValue],
    );

    const renderSuggestModalContainer = useImmutableCallback(
        (modalProps: {
            type: ESearchFormFieldName.FROM | ESearchFormFieldName.TO;
            hideModal(
                type: ESearchFormFieldName.FROM | ESearchFormFieldName.TO,
            ): void;
            triggerNode: React.ReactNode;
            componentNode: React.ReactNode;
        }) => {
            const {triggerNode, componentNode, type, hideModal} = modalProps;

            return (
                <SearchFormFieldModal
                    triggerNode={triggerNode}
                    componentNode={componentNode}
                    fieldType={type}
                    error={error}
                    buttonsAreVisible={false}
                    cancelButtonIsVisible
                    onHideModal={hideModal}
                    onCancelClick={handleCancelClick}
                    onButtonClick={handleModalButtonClick}
                    resetInputValue={resetInputValue}
                />
            );
        },
    );

    const preparedValue = useMemo(() => {
        return typeof selectedValue === 'boolean'
            ? false
            : getSuggestTitleAndDescription(selectedValue);
    }, [getSuggestTitleAndDescription, selectedValue]);

    return (
        <div className={cx('root', deviceMods('root', deviceType), className)}>
            <BaseSearchSuggest
                triggerClassName={triggerClassName}
                triggerFocusClassName={triggerFocusClassName}
                items={suggestItems}
                type={name}
                error={Boolean(error)}
                otherField={otherField}
                deviceType={deviceType}
                triggerRef={setFieldRefByName(name)}
                errorTriggerRef={setErrorFieldRefByName(name)}
                inputValue={inputValue}
                selectedValue={preparedValue}
                tabIndex={isExpanded ? 0 : -1}
                placeholder={placeholder}
                modalContainer={renderSuggestModalContainer}
                uniqueValueName={uniqueValueName}
                previousSuggestItems={previousSuggestItems}
                getItemTitleAndDescription={getSuggestTitleAndDescription}
                isModalView={deviceType.isMobile}
                isAutoClosable
                onHidePopup={handleHideSuggestDropdown}
                onFinishSelect={handleSuggestFinishSelect}
                onFocus={handleFocusSuggest}
                onBlur={handleBlurSuggest}
                onResetValue={handleResetValue}
                onSelectItem={handleSelectItem}
                onModalHistoryBack={handleOnModalHistoryBack}
                onChangeInputValue={handleChangeInputValue}
                renderSuggestItemTitle={renderSuggestItemTitle}
                renderSuggestItemDescription={renderSuggestItemDescription}
                renderSuggestItemIcon={renderSuggestItemIcon}
                defaultAutoSelectIndex={defaultAutoSelectIndex}
                isSuggestLoading={isSuggestLoading}
                sortItems={sortItems}
                onShowSuggests={onShowSuggests}
                validateOnMount={validateOnMount}
                {...prepareQaAttributes({
                    parent: rest,
                    current: `search-suggest-${name}`,
                })}
            />
        </div>
    );
};

export default typedReactMemo(Suggest);
