import {Component, ReactNode} from 'react';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/components';

/* Components */
import Button from 'components/Button/Button';
import ArrowLongRightIcon from 'icons/16/ArrowLongRight';

/* Init Styles */
import cx from './SearchFormFieldModal.scss';

/* Constants */
const DEFAULT_BUTTONS: ISearchFormFieldModalButton[] = [
    {type: 'complete', text: i18nBlock.searchModalDotComplete()},
    {type: 'next', text: i18nBlock.searchModalDotNext()},
];

const ROOT_QA = 'search-form-field-modal';

/* Components Types */
interface ISearchFormFieldModalButton {
    type: string;
    text: string;
}

interface ISearchFormFieldModalProps<T = string> {
    isLastControl?: boolean;
    buttonsAreVisible: boolean;
    cancelButtonIsVisible: boolean;
    fieldType: T;
    triggerNode: ReactNode;
    componentNode: ReactNode;
    error?: string;
    buttons: ISearchFormFieldModalButton[];
    onHideModal: (fieldType: T) => void;
    onCancelClick: (fieldType: T) => void;
    onButtonClick?: (params: {buttonType: string; fieldType: T}) => void;
    resetInputValue?: () => void;
}

class SearchFormFieldModal<T = string> extends Component<
    ISearchFormFieldModalProps<T>
> {
    static readonly defaultProps: Partial<ISearchFormFieldModalProps> = {
        buttons: DEFAULT_BUTTONS,
        buttonsAreVisible: true,
    };

    /* Handlers */
    private handleFooterButtonClick = (buttonType: string): void => {
        const {fieldType, onHideModal, onButtonClick} = this.props;

        if (onHideModal) {
            onHideModal(fieldType);
        }

        if (onButtonClick) {
            onButtonClick({buttonType, fieldType});
        }
    };

    private handleCancelButtonClick = (): void => {
        const {fieldType, onHideModal, onCancelClick, resetInputValue} =
            this.props;

        if (onHideModal) {
            onHideModal(fieldType);
        }

        if (onCancelClick) {
            onCancelClick(fieldType);
        }

        if (resetInputValue) {
            resetInputValue();
        }
    };

    /* Render */
    private renderHeader(): ReactNode {
        const {triggerNode, cancelButtonIsVisible} = this.props;

        return (
            <div className={cx('trigger')}>
                {cancelButtonIsVisible && (
                    <button
                        className={cx('cancelButton')}
                        type="button"
                        onClick={this.handleCancelButtonClick}
                        {...prepareQaAttributes({
                            parent: ROOT_QA,
                            current: 'cancel',
                        })}
                    >
                        <ArrowLongRightIcon
                            className={cx('backIcon')}
                            width={20}
                            height={64}
                        />
                    </button>
                )}
                {triggerNode}
            </div>
        );
    }

    private renderSearchFormComponent(): ReactNode {
        const {componentNode, buttonsAreVisible} = this.props;

        return (
            <div
                className={cx('component', {
                    component_withButtons: buttonsAreVisible,
                })}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'component',
                })}
            >
                {componentNode}
            </div>
        );
    }

    private renderButton = (
        button: ISearchFormFieldModalButton,
        index: number,
    ): ReactNode => {
        const {type, text} = button;

        return (
            <Button
                key={index}
                className={cx('button')}
                onClick={this.handleFooterButtonClick.bind(this, type)}
                size="l"
                width="max"
                theme="primary"
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: `button-${type}`,
                })}
            >
                {text}
            </Button>
        );
    };

    private renderButtons(): ReactNode {
        const {buttonsAreVisible, buttons, isLastControl} = this.props;

        if (!buttonsAreVisible) {
            return null;
        }

        return (
            <div className={cx('buttons')}>
                {buttons
                    .filter(({type}) => !(isLastControl && type === 'next'))
                    .map(this.renderButton)}
            </div>
        );
    }

    private renderErrorLabel(): ReactNode {
        const {error} = this.props;

        if (error) {
            return <div className={cx('errorTooltip')}>{error}</div>;
        }
    }

    private renderFooter(): ReactNode {
        return (
            <div
                className={cx('footer')}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'footer',
                })}
            >
                {this.renderErrorLabel()}
                {this.renderButtons()}
            </div>
        );
    }

    render(): ReactNode {
        return (
            <div
                className={cx('searchModal')}
                {...prepareQaAttributes(ROOT_QA)}
            >
                <div className={cx('content')}>
                    {this.renderHeader()}
                    {this.renderSearchFormComponent()}
                    {this.renderFooter()}
                </div>
            </div>
        );
    }
}

export default SearchFormFieldModal;
