import React, {PureComponent, FocusEvent} from 'react';
import _noop from 'lodash/noop';

import {IWithDeviceType} from 'types/withDeviceType';
import {IWithClassName} from 'types/withClassName';

import appData from 'utilities/appData/appData';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import SearchControl from 'components/SearchControl/SearchControl';
import CloseCircleIcon from 'icons/16/CloseCircle';

const RESET_ICON_SIZE_MOBILE = {
    width: 20,
    height: 20,
};
const RESET_ICON_SIZE_DESKTOP = {
    width: 18,
    height: 18,
};

import cx from './SearchInput.scss';

interface ISelectedValue {
    title: string;
    description?: string;
}

export interface ISearchInputProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    placeholder: string;
    onFocus: Function;
    onBlur: Function;
    onKeyDown: Function;
    onChange: Function;
    isActive: boolean;
    inputValue: string;
    canRenderResetIcon: boolean;
    selectedValue: false | ISelectedValue;
    tabIndex: number;
    hasError: boolean;
    onResetValue: Function;
    refs: Function[];
    rootRefs: Function[];
    focusOnMount: boolean;
    isMobile: boolean;
    focusClassName: string;
}

interface ISearchInputState {
    isFocused: boolean;
    isMounted: boolean;
}

class SearchInput extends PureComponent<ISearchInputProps, ISearchInputState> {
    private autoCompleteOffHackValue = 'off-' + appData.getByKey('uid');

    private _inputNode?: HTMLInputElement;

    static readonly defaultProps = {
        placeholder: '',
        onFocus: _noop,
        onBlur: _noop,
        onKeyDown: _noop,
        onChange: _noop,
        isActive: false,
        selectedValue: false,
        canRenderResetIcon: true,
        onResetValue: _noop,
        hasError: false,
        focusOnMount: false,
        className: '',
        focusClassName: '',
        isMobile: false,
        inputValue: '',
        refs: [],
        rootRefs: [],
        deviceType: {},
    };

    readonly state: ISearchInputState = {
        isFocused: false,
        isMounted: false,
    };

    componentDidMount(): void {
        this.checkFocus();

        const value = this.getInputNodeRef()?.value;

        // если в компонент уже что то ввели - обновляем состояние чтобы не потерять ввод пользователя
        // такое может произойти когда страница отдается через SSR и пользователь успел ввести что то до загрузки скриптов
        if (this.props.inputValue !== value) {
            this.props.onChange({target: {value}});
        }

        this.setState({...this.state, isMounted: true});
    }

    /* Refs manipulate */

    private setRootNodeRef = (rootNode: HTMLLabelElement): void => {
        const {rootRefs} = this.props;

        rootRefs.forEach(ref => ref(rootNode));
    };

    private getInputNodeRef = (): HTMLInputElement | undefined => {
        return this._inputNode;
    };

    private setInputNodeRef = (inputNode: HTMLInputElement): void => {
        const {refs} = this.props;

        this._inputNode = inputNode;

        refs.forEach(ref => ref(inputNode));
    };

    /* Input handlers */

    private handleKeyDown = (e: React.KeyboardEvent): void => {
        const {onKeyDown} = this.props;

        onKeyDown(e);
    };

    private handleChange = (e: React.ChangeEvent<HTMLInputElement>): void => {
        const {onChange} = this.props;

        onChange(e);
    };

    private handleFocus = (e: FocusEvent<HTMLInputElement>): void => {
        const {onFocus} = this.props;

        this.setState({isFocused: true});
        this.moveInputCursorToEnd(e.target);
        onFocus(e);
    };

    private handleBlur = (e: FocusEvent<HTMLInputElement>): void => {
        const {onBlur} = this.props;

        onBlur(e);
        this.setState({isFocused: false});
    };

    private handleClickReset = (): void => {
        const {onResetValue, deviceType} = this.props;

        onResetValue();

        // async(improve animation) focus input after reset input value [for IE only]
        if (deviceType.isIe) {
            requestAnimationFrame(() => {
                this.focusInput();
            });
        }
    };

    /* Actions */

    private checkFocus = (): void => {
        const {focusOnMount} = this.props;
        const inputNode = this.getInputNodeRef();

        if (focusOnMount) {
            this.focusInput();
        } else {
            setTimeout(() => {
                inputNode?.blur();
            }, 0);
        }
    };

    private focusInput = (): void => {
        const inputNode = this.getInputNodeRef();

        if (inputNode) {
            inputNode.focus();
        }
    };

    private moveInputCursorToEnd = (eventTarget: HTMLInputElement): void => {
        const {inputValue} = this.props;
        const inputValueLength = inputValue.length;

        eventTarget.selectionStart = inputValueLength;
        eventTarget.selectionEnd = inputValueLength;
    };

    /* Render */

    private renderSelectedValue(): React.ReactNode {
        const {isFocused} = this.state;
        const {selectedValue, isMobile, focusOnMount} = this.props;

        if (isFocused || !selectedValue || (focusOnMount && isMobile)) {
            return null;
        }

        const {title, description} = selectedValue;
        const hasDescription = Boolean(description);

        return (
            <div className={cx('selectedValue')}>
                <div
                    className={cx('selectedValueTitle')}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'title',
                    })}
                >
                    {title}
                </div>

                {hasDescription && (
                    <div className={cx('selectedValueDescription')}>
                        {description}
                    </div>
                )}
            </div>
        );
    }

    private renderResetIcon(): React.ReactNode {
        const {canRenderResetIcon, selectedValue, inputValue, isMobile} =
            this.props;
        const canRender = canRenderResetIcon && (selectedValue || inputValue);
        const iconSize = isMobile
            ? RESET_ICON_SIZE_MOBILE
            : RESET_ICON_SIZE_DESKTOP;

        return (
            canRender && (
                <div
                    className={cx('resetIcon')}
                    onClick={this.handleClickReset}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'reset',
                    })}
                >
                    <CloseCircleIcon {...iconSize} />
                </div>
            )
        );
    }

    private renderInput(): React.ReactNode {
        const {inputValue, tabIndex, placeholder} = this.props;
        const {isMounted} = this.state;
        const isEmptyInput = inputValue === '';

        const inputNode = (
            <input
                ref={this.setInputNodeRef}
                className={cx('input', {input_center: isEmptyInput})}
                type="text"
                value={inputValue}
                tabIndex={tabIndex}
                autoComplete={this.autoCompleteOffHackValue}
                onFocus={this.handleFocus}
                onBlur={this.handleBlur}
                onChange={this.handleChange}
                onKeyDown={this.handleKeyDown}
                // показываем нативный placeholder пока компонент не подлючился чтобы избежать ввода поверх placeholder'а
                placeholder={isMounted ? '' : placeholder}
                {...prepareQaAttributes({parent: this.props, current: 'input'})}
            />
        );

        return (
            <SearchControl
                placeholderClassName={cx('placeholder')}
                // скрываем placeholder пока компонент еще не подключен чтобы избежать ввода поверх placeholder'а
                placeholder={isMounted ? placeholder : ''}
                controlNode={inputNode}
                isEmpty={isEmptyInput}
            />
        );
    }

    render(): React.ReactNode {
        const {isActive, isMobile, hasError, className, focusClassName} =
            this.props;

        return (
            <label
                className={cx(
                    className,
                    isActive && focusClassName,
                    'searchInput',
                    {
                        searchInput_focus: isActive,
                        searchInput_error: hasError,
                        searchInput_mobile: isMobile,
                    },
                )}
                ref={this.setRootNodeRef}
                {...prepareQaAttributes(this.props)}
            >
                {this.renderInput()}
                {this.renderSelectedValue()}
                {this.renderResetIcon()}
            </label>
        );
    }
}

export default SearchInput;
