import React, {useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';

import IPrice from 'utilities/currency/PriceInterface';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import Button from 'components/Button/Button';
import ButtonLink from 'components/ButtonLink/ButtonLink';

import cx from './SearchSegmentBookInfo.scss';

interface ISearchSegmentBookInfoProps
    extends IWithClassName,
        IWithQaAttributes {
    price: IPrice;
    info: string;
    buttonText: string;
    bookUrl?: string;
    onClick?(): void;
}

const SearchSegmentBookInfo: React.FC<ISearchSegmentBookInfoProps> = props => {
    const {className, price, info, onClick, buttonText, bookUrl} = props;

    const deviceType = useDeviceType();

    const {isMobile} = deviceType;

    const bookButton = useMemo(() => {
        if (bookUrl) {
            return (
                <ButtonLink
                    className={cx('button')}
                    to={bookUrl}
                    size={isMobile ? 'l' : 'm'}
                    theme="primary"
                    onClick={onClick}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'actionButton',
                    })}
                >
                    {buttonText}
                </ButtonLink>
            );
        }

        return (
            <Button
                className={cx('button')}
                size={isMobile ? 'l' : 'm'}
                theme="primary"
                onClick={onClick}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'actionButton',
                })}
            >
                <Text color="primary">{buttonText}</Text>
            </Button>
        );
    }, [bookUrl, buttonText, isMobile, onClick, props]);

    return (
        <Flex
            className={cx('root', deviceMods('root', deviceType), className)}
            flexDirection={isMobile ? 'row' : 'column'}
            inline={isMobile}
            alignItems="center"
            nowrap={isMobile}
            between="2"
        >
            <Flex
                flexDirection="column"
                textColor="primary"
                alignItems={isMobile ? 'left' : 'center'}
            >
                <Price
                    className={cx('price')}
                    value={price.value}
                    currency={price.currency}
                    isRound
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'price',
                    })}
                />

                <Text
                    size="s"
                    color={isMobile ? 'secondary' : 'primary'}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'info',
                    })}
                >
                    {info}
                </Text>
            </Flex>

            {bookButton}
        </Flex>
    );
};

export default React.memo(SearchSegmentBookInfo);
