import React, {PureComponent, createElement, ReactNode} from 'react';
import _flow from 'lodash/flow';
import _noop from 'lodash/noop';
import _debounce from 'lodash/debounce';
import _delay from 'lodash/delay';
import _isEqual from 'lodash/isEqual';

import {
    ENTER_KEY_CODE,
    ARROW_UP_KEY_CODE,
    ARROW_DOWN_KEY_CODE,
} from 'constants/eventKeyCodes';

import {IWithDeviceType} from 'types/withDeviceType';
import {ISelectItemOptions} from 'components/SearchForm/types/ISelectItemOptions';
import EPopupDirection from 'components/Popup/types/EPopupDirection';
import {TOnShowSuggests} from 'components/SearchSuggest/types/TOnShowSuggests';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getItemsByPreviousSearch from './utilities/getItemsByPreviousSearch';
import {deviceMods} from 'utilities/stylesUtils';
import isNearbySuggest from './utilities/isNearbySuggest';
import isPersonalSuggest from './utilities/isPersonalSuggest';
import isPreviousSuggest from './utilities/isPreviousSuggest';

import * as i18nBlock from 'i18n/components';

import Dropdown, {
    IDropdownPopupParams,
    IDropdownSwitcherParams,
} from 'components/Dropdown/Dropdown';
import SearchInput, {
    ISearchInputProps,
} from 'components/SearchInput/SearchInput';
import {IPopupProps} from 'components/Popup/Popup';
import LocationIcon from 'icons/16/Location';
import Spinner from 'components/Spinner/Spinner';
import Text from 'components/Text/Text';
import HotelIcon from 'icons/16/Hotel';
import CrossSaleSearchSuggest from './components/CrossSaleSearchSuggest/CrossSaleSearchSuggest';
import PreviousSearchSuggest from './components/PreviousSearchSuggest/PreviousSearchSuggest';
import Flex from 'components/Flex/Flex';

import cx from './SearchSuggest.scss';

const DEFAULT_ACTIVE_ELEMENT_INDEX = -1;
const POPUP_DEBOUNCE_TIME = 50;
const SET_RENDER_PREVIOUS_SEARCH_DELAY = 50;
const MAX_PREVIOUS_COUNT = 3;
const MAX_ITEMS_COUNT = 5;

export interface ISuggestItemHandlers {
    onMouseDown: React.MouseEventHandler<HTMLInputElement>;
    onMouseEnter?: React.MouseEventHandler<HTMLInputElement>;
    onMouseLeave?: React.MouseEventHandler<HTMLInputElement>;
}

export interface ITitleAndDescription {
    title: string;
    description?: string;
}

export interface ISuggestItemRenderProps<SuggestItem> {
    item: SuggestItem;
    items: SuggestItem[];
    index: number;
    activeElementIndex?: number;
    isSuggestLoading?: boolean;
    input?: ISuggestItemHandlers;
}

export type TSelectedValue<SuggestItem> = SuggestItem | false;

export enum ESuggestSource {
    INPUT = 'INPUT',
    SUGGESTS = 'SUGGESTS',
    PRESETS = 'PRESETS',
    SEARCH_CONTEXT = 'SEARCH_CONTEXT',
    CROSS_SEARCH = 'CROSS_SEARCH',
}

export interface ISuggestValue<SuggestItem> {
    selectedValue: TSelectedValue<SuggestItem>;
    inputValue: string;
    source: ESuggestSource;
}

export interface IModalContainerProps<FieldType> {
    type: FieldType;
    triggerNode: React.ReactNode;
    componentNode: React.ReactNode;
    hideModal: (type: FieldType) => void;
}

export interface ISearchSuggestProps<SuggestItem, FieldType = string>
    extends IWithDeviceType,
        IWithQaAttributes {
    items: SuggestItem[];
    type: FieldType;
    inputValue: string;
    canRenderResetIcon?: boolean;
    isSuggestLoading?: boolean;
    defaultAutoSelectIndex?: number;
    getItemTitleAndDescription: (item: SuggestItem) => ITitleAndDescription;
    placeholder: string;
    tabIndex: number;
    selectedValue: false | ITitleAndDescription;
    hideEmptyList?: boolean;
    isAutoClosable?: boolean;
    onSelectItem: (
        item: TSelectedValue<SuggestItem>,
        options?: ISelectItemOptions,
    ) => void;
    onChangeInputValue: (value: string) => void;
    onFocus: (type: FieldType, inputValue: string) => void;
    onBlur: (type: FieldType, inputValue: string) => void;
    onHidePopup: (type: FieldType) => void;
    onResetValue: () => void;
    renderSuggestItem?: (
        props: ISuggestItemRenderProps<SuggestItem>,
    ) => React.ReactElement;
    renderSuggestItemTitle?: (item: SuggestItem) => ReactNode;
    renderSuggestItemDescription?: (item: SuggestItem) => ReactNode;
    renderSuggestItemIcon?: (item: SuggestItem) => ReactNode;
    triggerRef: (node: HTMLInputElement | HTMLButtonElement) => void;
    onFinishSelect: (
        type: FieldType,
        item: TSelectedValue<SuggestItem>,
    ) => void;
    error: boolean | string[];
    errorTriggerRef: (node: HTMLLabelElement | HTMLButtonElement) => void;
    onShowPopup: (type: FieldType) => void;
    onModalHistoryBack: (type: FieldType) => void;
    triggerClassName?: string;
    uniqueValueName: keyof SuggestItem;
    previousSearchTitle: string;
    previousSuggestItems: SuggestItem[];
    triggerFocusClassName: string;
    otherField: TSelectedValue<SuggestItem>;
    maxPreviousCount?: number;
    isModalView?: boolean;
    maxCount?: number;
    modalContainer: (
        props: IModalContainerProps<FieldType>,
    ) => React.ReactElement;

    sortItems?: (a: SuggestItem, b: SuggestItem) => number;
    /**
     * Вызывается при показе саджестов.
     * При изменении списка отображаемых саджестов вызывается снова.
     */
    onShowSuggests?: TOnShowSuggests<FieldType, SuggestItem>;
    validateOnMount?: boolean;
}

interface ISearchSuggestState {
    activeElementIndex: number;
    isHiddenInputValue: boolean;
    canRenderPreviousSearch: boolean;
    canHideSuggestList: boolean;
    showSuggestDropdown: boolean;
}

export interface IBaseSuggestItem {
    parentId?: string | boolean;
    isPreviousSearch?: boolean;
    groupName?: string;
}

class SearchSuggest<S extends IBaseSuggestItem, T> extends PureComponent<
    ISearchSuggestProps<S, T>,
    ISearchSuggestState
> {
    private dropdownInstance: Dropdown | null = null;
    private modalInputNode: HTMLInputElement | undefined;
    private inputNode: HTMLInputElement | undefined;
    private prevSuggests: S[] = [];

    static readonly defaultProps = {
        items: [],
        inputValue: '',
        defaultAutoSelectIndex: 0,
        error: false,
        selectedValue: false,
        hideEmptyList: true,
        isAutoClosable: false,
        onSelectItem: _noop,
        onChangeInputValue: _noop,
        onFocus: _noop,
        onBlur: _noop,
        onShowPopup: _noop,
        canRenderResetIcon: true,
        onResetValue: _noop,
        onHidePopup: _noop,
        triggerRef: _noop,
        errorTriggerRef: _noop,
        onFinishSelect: _noop,
        deviceType: {},
        otherField: false,
        previousSuggestItems: [],
        isModalView: false,
        previousSearchTitle: i18nBlock.searchSuggestDotPreviousSearchTitle(),
        uniqueValueName: 'id',
        triggerClassName: '',
        triggerFocusClassName: '',
    };

    readonly state: ISearchSuggestState = {
        activeElementIndex: DEFAULT_ACTIVE_ELEMENT_INDEX,
        canRenderPreviousSearch: true,
        canHideSuggestList: false,
        isHiddenInputValue: false,
        showSuggestDropdown: false,
    };

    constructor(props: ISearchSuggestProps<S, T>) {
        super(props);

        this.handleHidePopup = _debounce(
            this.handleHidePopup,
            POPUP_DEBOUNCE_TIME,
        );
    }

    componentDidMount(): void {
        if (!this.props.validateOnMount) {
            setTimeout(() => {
                this.hidePopup();
            });
        }
    }

    componentDidUpdate(): void {
        this.handleShowSuggests();
    }

    /* Refs manipulate */

    private setModalInputRef = (inputNode: HTMLInputElement): void => {
        this.modalInputNode = inputNode;
    };

    private getModalInputRef = (): HTMLInputElement | undefined => {
        return this.modalInputNode;
    };

    private setInputRef = (inputNode: HTMLInputElement | undefined): void => {
        this.inputNode = inputNode;
    };

    private getInputRef = (): HTMLInputElement | undefined => {
        return this.inputNode;
    };

    private setDropdownRef = (dropdownInstance: Dropdown | null): void => {
        this.dropdownInstance = dropdownInstance;
    };

    private getDropdownRef = (): Dropdown | null => {
        return this.dropdownInstance;
    };

    /* Refs instances actions */

    private showPopup = (): void => {
        const dropdownInstance = this.getDropdownRef();

        if (dropdownInstance) {
            dropdownInstance.showPopup();
        } else {
            // dropdown еще не инициализирован, пробуем открыть через проперти
            this.setState({...this.state, showSuggestDropdown: true});
        }
    };

    private hidePopup = (): void => {
        const dropdownInstance = this.getDropdownRef();

        if (dropdownInstance) {
            dropdownInstance.hidePopup();
        }
    };

    /* Previous Search Form */

    private setRenderPreviousSearchStatus(inputValue = ''): void {
        _delay(() => {
            this.setState(state => ({
                canRenderPreviousSearch:
                    state.isHiddenInputValue || inputValue === '',
            }));
        }, SET_RENDER_PREVIOUS_SEARCH_DELAY);
    }

    private getAllItems = (): S[] => {
        const {canRenderPreviousSearch} = this.state;
        const {
            items,
            uniqueValueName,
            previousSuggestItems,
            otherField,
            maxCount = MAX_ITEMS_COUNT,
            maxPreviousCount = MAX_PREVIOUS_COUNT,
            sortItems,
        } = this.props;

        return getItemsByPreviousSearch({
            items,
            maxCount,
            otherField,
            uniqueValueName,
            maxPreviousCount,
            previousSuggestItems,
            canRenderPreviousSearch,
            sortItems,
        });
    };

    /* Input field handlers */

    private resetActiveElementIndex(): void {
        this.setState({
            activeElementIndex: DEFAULT_ACTIVE_ELEMENT_INDEX,
        });
    }

    private handleChangeSearchInput = (
        e: React.ChangeEvent<HTMLInputElement>,
    ): void => {
        const {onChangeInputValue} = this.props;
        const inputValue = e.target.value;

        this.setState({isHiddenInputValue: false});
        this.showPopup();
        this.setRenderPreviousSearchStatus(inputValue);
        this.resetActiveElementIndex();

        onChangeInputValue(inputValue);
    };

    private handleResetValue = (): void => {
        const {onResetValue} = this.props;

        this.resetActiveElementIndex();
        this.setRenderPreviousSearchStatus();
        onResetValue();
    };

    private handleFocus = (
        e: React.FocusEvent<HTMLInputElement>,
    ): React.FocusEvent<HTMLInputElement> => {
        const {onFocus, type, inputValue, deviceType} = this.props;

        if (deviceType.isDesktop) {
            e.target.setSelectionRange(0, inputValue.length);
        }

        this.resetActiveElementIndex();

        onFocus(type, inputValue);

        return e;
    };

    private handleBlur = (): void => {
        const {onBlur, type, inputValue} = this.props;

        onBlur(type, inputValue);
    };

    private handleFocusModalInput = (
        e: React.FocusEvent<HTMLInputElement>,
    ): React.FocusEvent<HTMLInputElement> => {
        this.handleFocus(e);
        this.setState({
            canHideSuggestList: false,
            canRenderPreviousSearch: true,
            isHiddenInputValue: true,
        });

        return e;
    };

    private handleFinishSelect = (): void => {
        const inputNode = this.getInputRef();

        this.blurInputField(inputNode);
    };

    private handleFinishSelectForModalView = (): void => {
        const inputNode = this.getModalInputRef();

        this.blurInputField(inputNode);
    };

    /* Input Actions */

    private blurInputField = (
        inputNode: HTMLInputElement | undefined,
    ): void => {
        if (inputNode && typeof inputNode.blur === 'function') {
            inputNode.blur();
        }
    };

    /* Input field key handlers */

    private handleKeyDown = (e: React.KeyboardEvent): React.KeyboardEvent => {
        const {keyCode} = e;

        switch (keyCode) {
            case ARROW_UP_KEY_CODE: {
                this.handleUpActiveElement(e);

                break;
            }

            case ARROW_DOWN_KEY_CODE: {
                this.handleDownActiveElement(e);

                break;
            }

            case ENTER_KEY_CODE: {
                this.handleEnterPress(e);
            }
        }

        return e;
    };

    private handleUpActiveElement = (e: React.KeyboardEvent): void => {
        const {activeElementIndex} = this.state;
        const items = this.getAllItems();
        const lastElementIndex = items.length - 1;
        let currentActiveElementIndex = activeElementIndex - 1;

        if (activeElementIndex === 0) {
            currentActiveElementIndex = lastElementIndex;
        }

        this.setState({activeElementIndex: currentActiveElementIndex});
        this.preventDefaultAndStopPropagation(e);
    };

    private handleDownActiveElement = (e: React.KeyboardEvent): void => {
        const {activeElementIndex} = this.state;
        const items = this.getAllItems();
        const lastElementIndex = items.length - 1;
        let currentActiveElementIndex = activeElementIndex + 1;

        if (lastElementIndex === activeElementIndex) {
            currentActiveElementIndex = 0;
        }

        this.setState({activeElementIndex: currentActiveElementIndex});
        this.preventDefaultAndStopPropagation(e);
    };

    private handleEnterPress = (e: React.KeyboardEvent): void => {
        const {activeElementIndex} = this.state;

        if (activeElementIndex !== DEFAULT_ACTIVE_ELEMENT_INDEX) {
            this.handleSelectItem(activeElementIndex, e);
        }

        this.preventDefaultAndStopPropagation(e);
    };

    /* Suggest list handlers */

    private handleClickSuggestItem = (
        index: number,
        e: React.MouseEvent,
    ): void => {
        this.handleSelectItem(index, e);

        this.preventDefaultAndStopPropagation(e);
    };

    /**
     * Select suggest item by active index
     *
     * @param activeIndex - active suggest item
     * @param e - init event
     */
    private handleSelectItem = async (
        activeIndex: number,
        e: React.KeyboardEvent | React.MouseEvent,
    ): Promise<void> => {
        const {
            onSelectItem,
            onFinishSelect,
            type,
            isModalView,
            isAutoClosable,
        } = this.props;
        const items = this.getAllItems();
        const item = items[activeIndex];

        await onSelectItem(item, {isTrustedUser: e.isTrusted});

        if (isModalView) {
            this.handleFinishSelectForModalView();
            this.setState({canHideSuggestList: true});
        } else {
            this.handleFinishSelect();
            onFinishSelect(type, item);
        }

        if (!isModalView || isAutoClosable) {
            this.hidePopup();
        }
    };

    private handleMouseEnterSuggestItem = (index: number): void => {
        this.setState({activeElementIndex: index});
    };

    private handleMouseLeaveSuggestItem = (): void => {
        this.setState({activeElementIndex: DEFAULT_ACTIVE_ELEMENT_INDEX});
    };

    private handleShowSuggests(): void {
        const {onShowSuggests, type} = this.props;
        const suggestsForRender = this.getAllItems();

        if (
            this.dropdownInstance?.isVisible() &&
            !_isEqual(this.prevSuggests, suggestsForRender)
        ) {
            this.prevSuggests = suggestsForRender;

            onShowSuggests?.(type, suggestsForRender);
        }
    }

    private handleShowPopup = (): void => {
        const {onShowPopup, type, inputValue} = this.props;

        onShowPopup(type);
        this.handleShowSuggests();
        this.setRenderPreviousSearchStatus(inputValue);
    };

    private handleHidePopup = (): void => {
        const {
            type,
            items,
            inputValue,
            onHidePopup,
            onSelectItem,
            selectedValue,
            defaultAutoSelectIndex,
        } = this.props;

        const canAutoSelect =
            selectedValue === false && items.length !== 0 && inputValue !== '';

        if (canAutoSelect) {
            onSelectItem(items[defaultAutoSelectIndex || 0], {
                isManualClick: false,
                isUserInput: false,
            });
        }

        onHidePopup(type);
        this.setRenderPreviousSearchStatus(inputValue);
    };

    private getSuggestItemHandlers = (index: number): ISuggestItemHandlers => {
        const {
            deviceType: {isMobile},
        } = this.props;

        return {
            onMouseDown: this.handleClickSuggestItem.bind(this, index),
            ...(isMobile
                ? {
                      onMouseEnter: this.handleMouseEnterSuggestItem.bind(
                          this,
                          index,
                      ),
                      onMouseLeave: this.handleMouseLeaveSuggestItem.bind(
                          this,
                          index,
                      ),
                  }
                : {}),
        };
    };

    private getCommonTriggerProps = (): Pick<
        ISearchInputProps,
        | 'isMobile'
        | 'tabIndex'
        | 'placeholder'
        | 'selectedValue'
        | 'onResetValue'
        | 'onChange'
    > => {
        const {
            selectedValue,
            tabIndex,
            placeholder,
            deviceType: {isMobile},
        } = this.props;

        return {
            isMobile,
            tabIndex,
            placeholder,
            selectedValue,
            onResetValue: this.handleResetValue,
            onChange: this.handleChangeSearchInput,
        };
    };

    private preventDefaultAndStopPropagation(
        e: React.MouseEvent | React.KeyboardEvent,
    ): void {
        e.preventDefault();
        e.stopPropagation();
    }

    private getPopupProps = (): Partial<IPopupProps> => {
        const {isModalView} = this.props;

        if (isModalView) {
            return {};
        }

        return {
            directions: [EPopupDirection.BOTTOM_LEFT],
            plain: true,
            nonvisual: false,
        };
    };

    /* Modal View */

    private renderModalTriggerInput = (): React.ReactNode => {
        const {triggerClassName, inputValue} = this.props;
        const {isHiddenInputValue} = this.state;

        return (
            <SearchInput
                inputValue={isHiddenInputValue ? '' : inputValue}
                focusOnMount
                className={cx('modalTriggerInput', triggerClassName)}
                refs={[this.setModalInputRef]}
                onFocus={this.handleFocusModalInput}
                onKeyDown={this.handleKeyDown}
                canRenderResetIcon={!isHiddenInputValue}
                {...this.getCommonTriggerProps()}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'modal-trigger',
                })}
            />
        );
    };

    private renderModalSuggest = ({
        meta,
    }: IDropdownPopupParams<{}>): React.ReactElement | null => {
        const {modalContainer, type} = this.props;
        const {canHideSuggestList} = this.state;
        const {hidePopup, visible} = meta;

        if (!visible) {
            return null;
        }

        const componentNode = canHideSuggestList
            ? null
            : this.renderSuggestList();
        const triggerNode = this.renderModalTriggerInput();

        if (modalContainer) {
            return createElement(modalContainer, {
                type,
                triggerNode,
                componentNode,
                hideModal: hidePopup,
            });
        }

        return (
            <React.Fragment>
                {triggerNode}
                {componentNode}
            </React.Fragment>
        );
    };

    /* Dropdown View */

    private renderTriggerInput = ({
        input,
        meta,
    }: IDropdownSwitcherParams): React.ReactElement => {
        const {onFocus, onKeyDown, ref: dropdownRef} = input;
        const {visible} = meta;
        const {
            inputValue,
            error,
            triggerRef,
            errorTriggerRef,
            deviceType,
            triggerClassName,
            triggerFocusClassName,
        } = this.props;

        const {isDesktop} = deviceType;
        const handleKeyDown = _flow(this.handleKeyDown, onKeyDown);
        const handleFocus = _flow(this.handleFocus, onFocus);

        return (
            <SearchInput
                inputValue={inputValue}
                isActive={visible}
                deviceType={deviceType}
                refs={[triggerRef, this.setInputRef]}
                className={triggerClassName}
                focusClassName={triggerFocusClassName}
                rootRefs={[dropdownRef, errorTriggerRef]}
                onFocus={handleFocus}
                onBlur={this.handleBlur}
                onKeyDown={handleKeyDown}
                canRenderResetIcon={isDesktop}
                hasError={Boolean(error)}
                {...this.getCommonTriggerProps()}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'trigger',
                })}
            />
        );
    };

    private renderDafaultSuggestItemIcon = (item: S): ReactNode => {
        const {renderSuggestItemIcon} = this.props;

        if (renderSuggestItemIcon) {
            const icon = renderSuggestItemIcon(item);

            if (icon) return <div className={cx('icon')}>{icon}</div>;
        }

        return null;
    };

    private renderDefaultSuggestItem = (
        items: S[],
        item: S,
        index: number,
    ): React.ReactNode => {
        const {deviceType} = this.props;
        const {activeElementIndex} = this.state;
        const {
            previousSearchTitle,
            getItemTitleAndDescription,
            renderSuggestItemTitle,
            renderSuggestItemDescription,
            isSuggestLoading,
        } = this.props;

        const {parentId} = item;
        const {title, description} = getItemTitleAndDescription(item);

        const isActiveElement = activeElementIndex === index;
        const hasDescription = Boolean(description);
        const nearbySuggest = isNearbySuggest(item);

        const prevItem = index > 0 ? items[index - 1] : undefined;

        let groupName = '';

        if (nearbySuggest) {
            groupName = '';
        } else if (item.isPreviousSearch && !prevItem?.isPreviousSearch) {
            groupName = previousSearchTitle;
        } else if (
            item.groupName &&
            (!prevItem ||
                (prevItem.isPreviousSearch && !item.isPreviousSearch) ||
                prevItem.groupName !== item.groupName)
        ) {
            groupName = item.groupName;
        }

        const itemQa = prepareQaAttributes({
            key: index,
            parent: this.props,
            current: 'list-item',
        });

        if (isPersonalSuggest(item)) {
            return (
                <CrossSaleSearchSuggest
                    Icon={HotelIcon}
                    groupName={groupName}
                    isActiveElement={isActiveElement}
                    title={
                        renderSuggestItemTitle
                            ? renderSuggestItemTitle(item)
                            : title
                    }
                    description={
                        renderSuggestItemDescription
                            ? renderSuggestItemDescription(item)
                            : description
                    }
                    deviceType={deviceType}
                    suggestItemHandlers={this.getSuggestItemHandlers(index)}
                    key={index}
                    {...itemQa}
                />
            );
        }

        if (isPreviousSuggest(item) || item.isPreviousSearch) {
            return (
                <PreviousSearchSuggest
                    groupName={groupName}
                    isActiveElement={isActiveElement}
                    title={
                        renderSuggestItemTitle
                            ? renderSuggestItemTitle(item)
                            : title
                    }
                    deviceType={deviceType}
                    description={
                        renderSuggestItemDescription
                            ? renderSuggestItemDescription(item)
                            : description
                    }
                    suggestItemHandlers={this.getSuggestItemHandlers(index)}
                    key={index}
                    {...itemQa}
                />
            );
        }

        return (
            <React.Fragment key={index}>
                {groupName && (
                    <Text
                        className={cx('suggestListTitle')}
                        size="s-inset"
                        color="primary"
                        tag="div"
                    >
                        {groupName}
                    </Text>
                )}

                <Flex
                    key={index}
                    className={cx(
                        'suggestItem',
                        {
                            suggestItem_active: isActiveElement,
                            suggestItem_child: parentId,
                        },
                        deviceMods('suggestItem', deviceType),
                    )}
                    justifyContent="flex-start"
                    alignItems="center"
                    tagProps={{...this.getSuggestItemHandlers(index)}}
                    {...itemQa}
                >
                    {this.renderDafaultSuggestItemIcon(item)}

                    <Flex flexDirection="column">
                        <div
                            className={cx('suggestItemTitle', {
                                isSuggestWithGeo: nearbySuggest,
                            })}
                            {...prepareQaAttributes({
                                parent: itemQa,
                                current: 'title',
                            })}
                        >
                            {nearbySuggest && (
                                <div
                                    className={cx(
                                        'suggestItemTitle-locationIcon',
                                    )}
                                >
                                    <LocationIcon />
                                </div>
                            )}

                            {renderSuggestItemTitle
                                ? renderSuggestItemTitle(item)
                                : title}
                            {nearbySuggest && isSuggestLoading && (
                                <div className={cx('suggestItemTitle-spinner')}>
                                    <Spinner size="xxs" />
                                </div>
                            )}
                        </div>
                        {hasDescription && (
                            <div className={cx('suggestItemDescription')}>
                                {renderSuggestItemDescription
                                    ? renderSuggestItemDescription(item)
                                    : description}
                            </div>
                        )}
                    </Flex>
                </Flex>
            </React.Fragment>
        );
    };

    private renderSuggestItem = ({
        items,
        item,
        index,
        isSuggestLoading,
    }: ISuggestItemRenderProps<S>): React.ReactNode => {
        const {activeElementIndex} = this.state;
        const {renderSuggestItem} = this.props;

        if (renderSuggestItem) {
            return createElement(renderSuggestItem, {
                item,
                items,
                index,
                key: index,
                activeElementIndex,
                isSuggestLoading,
                input: this.getSuggestItemHandlers(index),
            });
        }

        return this.renderDefaultSuggestItem(items, item, index);
    };

    private renderSuggestList = (): React.ReactElement | null => {
        const {
            hideEmptyList,
            deviceType: {isMobile},
            isModalView,
            isSuggestLoading,
        } = this.props;
        const items = this.getAllItems();

        if (items.length === 0 && hideEmptyList) {
            return null;
        }

        return (
            <div
                className={cx('suggestList', {
                    suggestList_mobile: isMobile,
                    suggestList_modal: isModalView,
                })}
                {...prepareQaAttributes({parent: this.props, current: 'list'})}
            >
                {items.map((item, index) =>
                    this.renderSuggestItem({
                        items,
                        item,
                        index,
                        isSuggestLoading,
                    }),
                )}
            </div>
        );
    };

    private handleModalHistoryBack = (): void => {
        const {onModalHistoryBack, type} = this.props;

        onModalHistoryBack(type);
    };

    render(): React.ReactNode {
        const {isModalView} = this.props;
        const {showSuggestDropdown} = this.state;
        const popupProps = this.getPopupProps();

        return (
            <Dropdown
                ref={this.setDropdownRef}
                popupProps={popupProps}
                onHidePopup={this.handleHidePopup}
                onModalHistoryBack={this.handleModalHistoryBack}
                onShowPopup={this.handleShowPopup}
                popupClassName={cx('popupContainer')}
                popupComponent={
                    isModalView
                        ? this.renderModalSuggest
                        : this.renderSuggestList
                }
                switcherComponent={this.renderTriggerInput}
                isModalView={isModalView}
                visible={showSuggestDropdown}
            />
        );
    }
}

export default SearchSuggest;
