import {PureComponent, ReactNode} from 'react';

import {
    ETextBlocksType,
    ISectionTextBlock,
    TSectionTextChildBlock,
    IGeoLinkGroupBlock,
    ISpoilerBlock,
    ITextBlock,
    ISubSectionTextBlock,
    TLinkTextBlock,
} from 'types/common/seoPages/ITextBlocks';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';

import Heading from 'components/Heading/Heading';
import TextBlock from './components/TextBlock/TextBlock';
import SpoilerBlock from './components/SpoilerBlock/SpoilerBlock';
import GeoLinkGroupBlock from './components/GeoLinkGroupBlock/GeoLinkGroupBlock';
import Box, {TBoxSizes} from 'components/Box/Box';

import cx from './SectionTextBlock.scss';

interface ISectionTextBlockProps extends IWithClassName, IWithDeviceType {
    block: ISectionTextBlock;
    betweenItems?: TBoxSizes;

    onLinkClick?(linkBlock: TLinkTextBlock): void;
}

class SectionTextBlock extends PureComponent<ISectionTextBlockProps> {
    private renderGeoLinkGroupBlock(
        block: IGeoLinkGroupBlock,
        index: number,
    ): ReactNode {
        const {onLinkClick} = this.props;

        return (
            <GeoLinkGroupBlock
                key={index}
                className={cx('geoLinkGroupBlock')}
                block={block}
                onLinkClick={onLinkClick}
            />
        );
    }

    private renderSpoilerBlock(block: ISpoilerBlock, index: number): ReactNode {
        const {onLinkClick} = this.props;

        return (
            <SpoilerBlock
                key={index}
                className={cx('spoilerBlock')}
                block={block}
                initVisibleDescriptionState
                onLinkClick={onLinkClick}
            />
        );
    }

    private renderTextBlock(block: ITextBlock, index: number): ReactNode {
        const {onLinkClick} = this.props;

        return (
            <TextBlock key={index} block={block} onLinkClick={onLinkClick} />
        );
    }

    private renderSubSectionTextBlock(
        block: ISubSectionTextBlock,
        index: number,
    ): ReactNode {
        const {
            data: {title},
            children,
        } = block;

        return (
            <div key={index} className={cx('subSection')}>
                {Boolean(title) && (
                    <Box below={3}>
                        <Heading level="3" className={cx('subTitle')}>
                            {title}
                        </Heading>
                    </Box>
                )}
                {children.map(this.renderSectionChildBlock)}
            </div>
        );
    }

    private renderSectionChildBlock = (
        block: TSectionTextChildBlock,
        index: number,
    ): ReactNode => {
        switch (block.type) {
            case ETextBlocksType.subSectionTextBlock: {
                return this.renderSubSectionTextBlock(block, index);
            }

            case ETextBlocksType.spoilerBlock: {
                return this.renderSpoilerBlock(block, index);
            }

            case ETextBlocksType.geoLinkGroupBlock: {
                return this.renderGeoLinkGroupBlock(block, index);
            }

            case ETextBlocksType.textBlock: {
                return this.renderTextBlock(block, index);
            }

            default: {
                return null;
            }
        }
    };

    render(): ReactNode {
        const {className, block, betweenItems = 0, deviceType} = this.props;
        const {
            data: {title, text},
            children = [],
        } = block;

        return (
            <div className={className}>
                {Boolean(title) && (
                    <Box below={deviceType.isDesktop ? 5 : 4}>
                        <Heading level="2">{title}</Heading>
                    </Box>
                )}

                {Boolean(text) && (
                    <Box below={deviceType.isDesktop ? 5 : 4} textSize="m">
                        {text}
                    </Box>
                )}

                <Box between={betweenItems}>
                    {children.map(this.renderSectionChildBlock)}
                </Box>
            </div>
        );
    }
}

export default SectionTextBlock;
