import React, {useCallback} from 'react';

import {IWithClassName} from 'types/withClassName';
import {TLinkTextBlock} from 'types/common/seoPages/ITextBlocks';

import getLinkUrlByBlockType from './utilities/getLinkUrlByBlockType';
import {useQueryByLocation} from 'utilities/hooks/useQueryByLocation';

import Link, {TLinkThemeType} from 'components/Link/Link';
import {TTextSize} from 'components/Text/Text';

import cx from './LinkTextBlock.scss';

export interface ILinkTextBlockProps extends IWithClassName {
    block: TLinkTextBlock;
    theme?: TLinkThemeType;
    size?: TTextSize;

    onClick?(linkBlock: TLinkTextBlock): void;
}

const LinkTextBlock: React.FC<ILinkTextBlockProps> = ({
    block,
    className,
    theme = 'normal',
    size = 'm',
    onClick,
}) => {
    const {
        data: {text},
    } = block;

    const queryByLocation = useQueryByLocation();
    const linkUrlByBlockType = getLinkUrlByBlockType(block, queryByLocation);

    const onLinkClick = useCallback(() => {
        onClick?.(block);
    }, [block, onClick]);

    if (!linkUrlByBlockType) {
        return null;
    }

    return (
        <Link
            className={cx('link', `link_size_${size}`, className)}
            theme={theme}
            target="_blank"
            rel="noopener noreferrer"
            url={linkUrlByBlockType}
            onClick={onLinkClick}
        >
            {text}
        </Link>
    );
};

export default LinkTextBlock;
