import React from 'react';

import {IWithClassName} from 'types/withClassName';
import {
    ETextBlocksType,
    EPlainTextStyle,
    ITextBlock,
    TLinkTextBlock,
} from 'types/common/seoPages/ITextBlocks';

import Price from 'components/Price/Price';
import Box, {TBoxSizes} from 'components/Box/Box';
import {TTextSize} from 'components/Text/Text';

import LinkTextBlock from '../LinkTextBlock/LinkTextBlock';

import cx from './TextBlock.scss';

export interface ITextBlockProps extends IWithClassName {
    block: ITextBlock;
    between?: TBoxSizes;
    textSize?: TTextSize;

    onLinkClick?(linkBlock: TLinkTextBlock): void;
}

const TextBlock: React.FC<ITextBlockProps> = props => {
    const {
        block: {children},
        between = 4,
        className,
        textSize = 'm',
        onLinkClick,
    } = props;

    return (
        <Box
            className={cx(className, 'textBlock')}
            textSize={textSize}
            between={between}
        >
            {children.map((block, index: number) => {
                switch (block.type) {
                    case ETextBlocksType.plainTextBlock: {
                        const {
                            data: {text, styles},
                        } = block;

                        return (
                            <span
                                key={index}
                                className={cx('plainText', {
                                    plainText_isBold:
                                        styles &&
                                        styles.includes(EPlainTextStyle.bold),
                                    plainText_isItalic:
                                        styles &&
                                        styles.includes(EPlainTextStyle.italic),
                                })}
                            >
                                {text}
                            </span>
                        );
                    }

                    case ETextBlocksType.priceTextBlock: {
                        const {
                            data: {price},
                        } = block;
                        const {value, currency} = price;

                        return (
                            <Price
                                key={index}
                                className={cx('price')}
                                value={value}
                                currency={currency}
                            />
                        );
                    }

                    case ETextBlocksType.regionLinkBlock:
                    case ETextBlocksType.searchHotelsLinkBlock:
                    case ETextBlocksType.hotelLinkBlock:
                    case ETextBlocksType.searchAviaLinkBlock:
                    case ETextBlocksType.routeLinkBlock:
                    case ETextBlocksType.externalLinkBlock: {
                        return (
                            <LinkTextBlock
                                key={index}
                                block={block}
                                size={textSize}
                                onClick={onLinkClick}
                            />
                        );
                    }

                    default: {
                        return null;
                    }
                }
            })}
        </Box>
    );
};

export default TextBlock;
