import {useState} from 'react';
import {Meta} from '@storybook/react/types-6-0';
import {noop} from 'lodash';

import {TStorybookFunctionComponent} from 'storybook/types';

import Box from 'components/Box/Box';
import CheckButton from 'components/CheckButton/CheckButton';
import StyleguidistPropTable from 'components/StyleguidistPropTable/StyleguidistPropTable';
import StyleguidistSizeTable from 'components/StyleguidistSizeTable/StyleguidistSizeTable';

import Select, {ISelectProps} from './Select';

export default {
    title: 'Example/Select',
    component: Select,
} as Meta;

const OPTIONS = [
    {value: 'a', data: 'Каждый'},
    {value: 'b', data: 'Охотник'},
    {value: 'c', data: 'Желает'},
    {value: 'd', data: 'Знать'},
    {value: 'e', data: 'Где', disabled: true},
    {value: 'f', data: 'Сидит'},
    {value: 'g', data: 'Фазан', disabled: true},
];

const LONG_OPTION = {
    value: 'h',
    data: 'И его мега-крутой друг с очень сложным именем',
};

const MANY_OPTIONS = Array.from(Array(1000), (_, idx) => ({
    value: idx,
    data: idx,
}));

export const State: TStorybookFunctionComponent = () => {
    const [values, setValues] = useState({
        secondary: '',
        outlined: '',
        maxWidth: '',
    });

    return (
        <StyleguidistPropTable
            props={[
                {
                    name: 'secondary',
                    props: {theme: 'secondary'},
                    comment: 'Default',
                    example: (
                        <Select
                            value={values.secondary}
                            options={OPTIONS}
                            placeholder="—"
                            onChange={(value): void =>
                                setValues({...values, secondary: value})
                            }
                        />
                    ),
                },
                {
                    name: 'outlined',
                    props: {theme: 'outlined'},
                    example: (
                        <Select
                            value={values.outlined}
                            theme="outlined"
                            options={OPTIONS}
                            placeholder="—"
                            onChange={(value): void =>
                                setValues({...values, outlined: value})
                            }
                        />
                    ),
                },
                {
                    name: 'disabled',
                    props: {disabled: true},
                    example: (
                        <Select
                            options={OPTIONS}
                            placeholder="—"
                            disabled
                            onChange={noop}
                        />
                    ),
                },
                {
                    name: 'disabled outlined',
                    props: {theme: 'outlined', disabled: true},
                    example: (
                        <Select
                            theme="outlined"
                            options={OPTIONS}
                            placeholder="—"
                            disabled
                            onChange={noop}
                        />
                    ),
                },
                {
                    name: 'max width',
                    props: {width: 'max'},
                    example: (
                        <div style={{width: 200}}>
                            <Select
                                value={values.maxWidth}
                                options={OPTIONS}
                                onChange={(value): void =>
                                    setValues({...values, maxWidth: value})
                                }
                                placeholder="—"
                                width="max"
                            />
                        </div>
                    ),
                },
            ]}
        />
    );
};

export const Sizes: TStorybookFunctionComponent = () => {
    return (
        <StyleguidistSizeTable
            sizes={[
                's' as const,
                'm-inset' as const,
                'm' as const,
                'l' as const,
            ].map(size => ({
                name: size,
                example: (
                    <Select
                        size={size}
                        onChange={noop}
                        value="a"
                        options={[...OPTIONS, LONG_OPTION]}
                        placeholder="—"
                    />
                ),
            }))}
        />
    );
};

export const WithVerticalScroll: TStorybookFunctionComponent = () => (
    <Select
        size="m"
        onChange={noop}
        options={MANY_OPTIONS}
        menuMaxHeight={300}
        placeholder="—"
    />
);

export const TriggerAndPopupEqualWidth: TStorybookFunctionComponent = () => {
    const [width, setWidth] = useState<ISelectProps<any, any>['width']>('auto');
    const [isVirtualized, setVirtualized] = useState(false);
    const [value, setValue] = useState(0);

    return (
        <Box between="2" inline nowrap>
            <CheckButton
                checked={width === 'max'}
                onClick={(): void => setWidth(width === 'max' ? 'auto' : 'max')}
            >
                Width Max
            </CheckButton>
            <CheckButton
                checked={isVirtualized}
                onClick={(): void => setVirtualized(!isVirtualized)}
            >
                Virtualized
            </CheckButton>
            <Select
                size="m-inset"
                onChange={setValue}
                value={value}
                options={MANY_OPTIONS}
                menuMaxHeight={300}
                width={width}
                virtualized={isVirtualized}
            />
        </Box>
    );
};
