import React, {
    ComponentType,
    useCallback,
    useEffect,
    useMemo,
    useRef,
    useState,
    MutableRefObject,
} from 'react';
import VirtualList, {ItemInfo} from 'react-tiny-virtual-list';
import {isUndefined, omitBy} from 'lodash';

import EPopupDirection from 'components/Popup/types/EPopupDirection';
import {IWithClassName} from 'types/withClassName';
import {IIconProps} from 'icons/types/icon';

import scrollContainerToNode from 'utilities/dom/scrollContainerToNode';
import {useEventListener} from 'utilities/hooks/useEventListener';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import {IPopupProps} from 'components/Popup/Popup';
import Dropdown from 'components/Dropdown/Dropdown';
import Button, {
    DEFAULT_SIZE,
    IButtonProps,
    TButtonTheme,
} from 'components/Button/Button';
import Separator from 'components/Separator/Separator';
import IncludeIcon from 'icons/16/Include';
import SelectArrow from 'icons/16/SelectArrow';
import Switcher, {
    IRenderButtonOptions,
    ISwitcherOwnProps,
} from 'components/Select/components/Switcher/Switcher';
import ArrowBottom from 'icons/16/ArrowBottom';

import cx from './Select.scss';

const VERTICAL_PADDING = 4;
const SCROLL_PADDING = 10;

const SEPARATOR = Symbol('separator');

const POPUP_PROPS: Partial<IPopupProps> = {
    directions: [
        EPopupDirection.BOTTOM_LEFT,
        EPopupDirection.BOTTOM_RIGHT,
        EPopupDirection.TOP_LEFT,
        EPopupDirection.TOP_RIGHT,
    ],
    gap: 'small',
};

export interface ISelectOption<Value, Data> extends IWithQaAttributes {
    value: Value;
    disabled?: boolean;
    data: Data;
}

export interface ISelectProps<Value, Data>
    extends IWithClassName,
        IWithQaAttributes {
    /**
     * Выбранное значение
     */
    value?: Value;

    /**
     * Размер кнопки селекта
     */
    size?: IButtonProps['size'];

    /**
     * Тема кнопки селекта
     */
    theme?: TButtonTheme;

    /**
     * Плейсхолдер будет отображён, если значение не будет выбрано
     */
    placeholder?: React.ReactNode;

    /**
     * Список элементов меню
     */
    options: (ISelectOption<Value, Data> | typeof SEPARATOR)[];

    /**
     * Максимальная высота меню
     */
    menuMaxHeight?: number;

    /**
     * Ширина меню:
     * - auto - ширина не зафиксирована
     * - fixed - ширина соотвествует ширине кнопки триггера
     */
    menuWidth?: 'auto' | 'fixed';

    /**
     * Ширина кнопки селекта
     */
    width?: IButtonProps['width'];

    /**
     * Ссылка на дом-ноду кнопки селекта
     */
    innerRef?: IButtonProps['innerRef'];
    listRef?: MutableRefObject<HTMLElement | null>;
    /**
     * Метод отрисовки элемента меню
     *
     * @param option - элемент из списка options
     * @param idx - индекс элемента
     */
    renderItem?(
        option: ISelectOption<Value, Data>,
        idx: number,
    ): React.ReactNode;
    /**
     * Метод отрисовки значения в кнопке
     *
     * @param option - элемент из списка options, который соответствует
     * переданному **value**. Если такого элемента нет, то будет вставлен
     * **placeholder**
     */
    renderValue?(option: ISelectOption<Value, Data>): React.ReactNode;

    name?: string;

    id?: string;

    /**
     * Отображает состояние ошибки
     */
    error?: boolean;

    /**
     * Включает виртуализацию списка
     */
    virtualized?: boolean;

    /**
     * Высота элемента для списка с виртуализацией
     */
    itemHeight?: number | ((index: number) => number);

    /**
     * Ширина меню для списка с виртуализацией
     */
    virtualizedMenuWidth?: number;

    scope?: React.RefObject<HTMLElement> | 'inplace';

    disabled?: boolean;

    fixPopupInSpring?: boolean;

    /**
     * Колбек, который срабатывает на выбор значения из селекта
     */
    onChange(value: Value, data: Data): void;

    onFocus?(): void;
    onBlur?(): void;
    onShowPopup?(): void;

    menuClassName?: string;

    /**
     * Флаг, включающий переворачиваемую стрелку вместо иконки по-умолчанию
     * TODO После прохода по кейсам использования заменить иконку по-умолчанию
     */
    adaptiveArrowIcon?: boolean;
    icon?: ComponentType<IIconProps>;
}

function Select<Value, Data>(
    props: ISelectProps<Value, Data>,
): React.ReactElement {
    const {
        className,
        menuClassName,
        error,
        id,
        innerRef,
        itemHeight = 32,
        menuMaxHeight = 400,
        menuWidth,
        name,
        options,
        placeholder,
        renderItem = defaultRenderItem,
        renderValue = defaultRenderValue,
        scope,
        size = DEFAULT_SIZE,
        theme,
        value,
        virtualized,
        virtualizedMenuWidth,
        width,
        disabled,
        onChange,
        onFocus,
        onBlur,
        adaptiveArrowIcon,
        onShowPopup,
        fixPopupInSpring = false,
        icon,
        listRef,
        ...rest
    } = props;

    const selectedItemIdx = options.findIndex(
        option => option !== SEPARATOR && option.value === value,
    );
    const rootQa = getQa(rest);
    const [hovered, setHovered] = useState(selectedItemIdx);
    const [minMenuWidth, setMinMenuWidth] = useState(0);

    const triggerRef = useRef<HTMLLabelElement>(null);
    const virtualListRef = useRef<VirtualList>(null);
    const dropdownRef = useRef<Dropdown<ISwitcherOwnProps>>(null);
    const menuContainerRef = useRef<HTMLDivElement>(null);
    const hoveredRef = useRef<HTMLButtonElement>(null);

    const popupProps = useMemo(() => {
        return omitBy({...POPUP_PROPS, fixPopupInSpring, scope}, isUndefined);
    }, [fixPopupInSpring, scope]);

    const updateMinMenuWidth = useCallback(() => {
        // Wait before all changes applied
        requestAnimationFrame(() => {
            const node = triggerRef.current;

            if (!node) {
                return;
            }

            setMinMenuWidth(node.clientWidth);
        });
    }, []);

    useEventListener('resize', updateMinMenuWidth);
    useEffect(() => updateMinMenuWidth());

    const lastHovered = useRef(hovered);

    lastHovered.current = hovered;

    const scrollToHoveredNode = useCallback(() => {
        const container = menuContainerRef.current;

        if (!container) {
            return;
        }

        if (virtualized && virtualListRef.current) {
            const offset = virtualListRef.current.state.offset;
            const menuHeight = menuMaxHeight;
            const vList = virtualListRef.current;
            const hoveredOffset = vList.getOffsetForIndex(lastHovered.current);
            const height =
                typeof itemHeight === 'function'
                    ? itemHeight(lastHovered.current)
                    : itemHeight;

            if (offset > hoveredOffset) {
                vList.scrollTo(hoveredOffset);
            } else if (hoveredOffset + height > offset + menuHeight) {
                vList.scrollTo(
                    hoveredOffset - menuHeight + height + SCROLL_PADDING,
                );
            }

            return;
        }

        const node = hoveredRef.current;

        if (node instanceof HTMLElement) {
            scrollContainerToNode(container, node, {
                verticalPadding: VERTICAL_PADDING,
            });
        }
    }, [itemHeight, menuMaxHeight, virtualized]);

    const handleHover = useCallback(
        ({currentTarget}: React.MouseEvent<HTMLElement>) => {
            const idx = Number(currentTarget.dataset.index);

            if (!isNaN(idx)) {
                setHovered(idx);
            }
        },
        [],
    );

    const select = useCallback(
        (idx: number) => {
            const option = options[idx];

            if (option !== SEPARATOR) {
                onChange(option.value, option.data);
            }
        },
        [onChange, options],
    );

    const handleMenuItemClick = useCallback(
        (event: React.MouseEvent<HTMLButtonElement>) => {
            const idx = Number(event.currentTarget.dataset.index);

            if (isNaN(idx)) {
                return;
            }

            const option = options[idx];

            if (option !== SEPARATOR) {
                onChange(option.value, option.data);
            }

            if (dropdownRef.current) {
                dropdownRef.current.hidePopup();
            }
        },
        [onChange, options],
    );

    const handleShowPopup = useCallback(() => {
        setHovered(selectedItemIdx);
        Promise.resolve().then(scrollToHoveredNode);
        onShowPopup?.();
    }, [scrollToHoveredNode, selectedItemIdx, onShowPopup]);

    const selectedOption = options.find(
        option => option !== SEPARATOR && option.value === value,
    );

    const hoverUpper = useCallback(() => {
        setHovered((idx: number): number => {
            let nextIdx = idx - 1;
            let option = options[nextIdx];

            while (option && (option === SEPARATOR || option.disabled)) {
                nextIdx--;
                option = options[nextIdx];
            }

            return Math.max(nextIdx, -1);
        });

        Promise.resolve().then(scrollToHoveredNode);
    }, [options, scrollToHoveredNode]);

    const hoverLower = useCallback(() => {
        setHovered((idx: number): number => {
            let nextIdx = idx + 1;
            let option = options[nextIdx];

            while (option && (option === SEPARATOR || option.disabled)) {
                nextIdx++;
                option = options[nextIdx];
            }

            if (!option || option.disabled) {
                return idx;
            }

            return nextIdx;
        });

        Promise.resolve().then(scrollToHoveredNode);
    }, [options, scrollToHoveredNode]);

    const selectHovered = useCallback(() => {
        select(hovered);
    }, [hovered, select]);

    const renderButton = useCallback(
        ({
            onClick,
            onKeyDown,
            onFocus,
            onBlur,
            visible,
        }: IRenderButtonOptions): React.ReactNode => {
            const Icon = icon || SelectArrow;
            const arrowIconNode = adaptiveArrowIcon ? (
                <ArrowBottom
                    className={cx(
                        'icon',
                        'arrow_icon',
                        `icon_size_${size}`,
                        visible && 'arrow_icon_open',
                    )}
                />
            ) : null;

            return (
                <Button
                    state={error ? 'error' : undefined}
                    size={size}
                    theme={theme}
                    width="max"
                    innerRef={innerRef}
                    name={name}
                    id={id}
                    disabled={disabled}
                    onClick={onClick}
                    onFocus={onFocus}
                    onBlur={onBlur}
                    onKeyDown={onKeyDown}
                    {...prepareQaAttributes({
                        parent: rootQa,
                        current: 'trigger',
                    })}
                >
                    <div className={cx('label')}>
                        <span
                            className={cx('label-text')}
                            {...prepareQaAttributes({
                                parent: rootQa,
                                current: 'labelText',
                            })}
                        >
                            {selectedOption &&
                            typeof selectedOption === 'object'
                                ? renderValue({...selectedOption, disabled})
                                : placeholder}
                        </span>

                        {arrowIconNode || (
                            <Icon className={cx('icon', `icon_size_${size}`)} />
                        )}
                    </div>
                </Button>
            );
        },
        [
            disabled,
            error,
            id,
            innerRef,
            name,
            placeholder,
            renderValue,
            selectedOption,
            size,
            adaptiveArrowIcon,
            theme,
            rootQa,
            icon,
        ],
    );

    const switcherProps = useMemo<ISwitcherOwnProps>(() => {
        return {
            className: cx(
                'trigger-wrapper',
                width && `trigger-wrapper_width_${width}`,
                {'trigger-wrapper_disabled': disabled},
                className,
            ),
            labelRef: triggerRef,
            renderButton,
            hoverUpper,
            hoverLower,
            selectHovered,
            onFocus,
            onBlur,
        };
    }, [
        className,
        disabled,
        hoverLower,
        hoverUpper,
        onBlur,
        onFocus,
        renderButton,
        selectHovered,
        width,
    ]);

    const getVirtualizedMenuHeight = useCallback((): number => {
        if (typeof itemHeight === 'function') {
            let height = 0;

            for (let i = 0; i < options.length; i++) {
                height += itemHeight(i);

                if (height > menuMaxHeight) {
                    return menuMaxHeight;
                }
            }

            return height;
        }

        return Math.min(menuMaxHeight, itemHeight * options.length);
    }, [itemHeight, menuMaxHeight, options.length]);

    const getItemHeight = useCallback(
        (idx: number): number => {
            const option = options[idx];

            if (option === SEPARATOR) {
                return 17;
            }

            if (typeof itemHeight === 'function') {
                return itemHeight(idx);
            }

            return itemHeight;
        },
        [itemHeight, options],
    );

    const renderOption = useCallback(
        (
            option: ISelectOption<Value, Data> | typeof SEPARATOR,
            idx: number,
            style?: React.CSSProperties,
        ): JSX.Element => {
            if (option === SEPARATOR) {
                return (
                    <div
                        key={idx}
                        className={cx(
                            'menu-list-item',
                            'menu-list-item__separator',
                        )}
                        style={style}
                    >
                        <Separator />
                    </div>
                );
            }

            return (
                <div
                    key={idx}
                    className={cx(
                        'menu-list-item',
                        hovered === idx && 'menu-item_active',
                    )}
                    style={style}
                >
                    <div className={cx('menu-list-item__offset')}>
                        {value === option.value && <IncludeIcon />}
                    </div>
                    <button
                        type="button"
                        ref={hovered === idx ? hoveredRef : undefined}
                        tabIndex={-1}
                        onClick={handleMenuItemClick}
                        disabled={option.disabled}
                        onMouseOver={handleHover}
                        data-index={idx}
                        className={cx('menu-item')}
                        {...prepareQaAttributes({
                            key: String(option.value),
                            parent: rootQa,
                            current: 'option',
                        })}
                    >
                        {renderItem(option, idx)}
                    </button>
                </div>
            );
        },
        [handleHover, handleMenuItemClick, hovered, renderItem, value, rootQa],
    );

    const renderVirtualListItem = useCallback(
        ({index, style}: ItemInfo): React.ReactNode => {
            return renderOption(options[index], index, style);
        },
        [options, renderOption],
    );

    const renderPopup = useCallback((): React.ReactNode => {
        const width = menuWidth === 'auto' ? 'auto' : minMenuWidth;

        return (
            <div
                className={cx('menu', menuClassName)}
                style={virtualized ? {} : {maxHeight: menuMaxHeight, width}}
                ref={menuContainerRef}
                {...prepareQaAttributes({
                    parent: rootQa,
                    current: 'optionsPopup',
                })}
            >
                {virtualized ? (
                    <VirtualList
                        itemCount={options.length}
                        itemSize={getItemHeight}
                        renderItem={renderVirtualListItem}
                        height={getVirtualizedMenuHeight()}
                        width={virtualizedMenuWidth || minMenuWidth}
                        ref={virtualListRef}
                    />
                ) : (
                    options.map((option, index) => renderOption(option, index))
                )}
            </div>
        );
    }, [
        getItemHeight,
        getVirtualizedMenuHeight,
        menuMaxHeight,
        menuWidth,
        minMenuWidth,
        options,
        renderOption,
        renderVirtualListItem,
        virtualized,
        virtualizedMenuWidth,
        rootQa,
    ]);

    return (
        <Dropdown<ISwitcherOwnProps>
            ref={dropdownRef}
            popupRef={listRef}
            switcherComponent={Switcher}
            switcherComponentProps={switcherProps}
            renderPopup={renderPopup}
            popupProps={popupProps}
            onShowPopup={handleShowPopup}
            popupMotionless
        />
    );
}

Select.Separator = SEPARATOR;

function defaultRenderValue(
    option: ISelectOption<unknown, any>,
): React.ReactNode {
    return option.data;
}

function defaultRenderItem(
    option: ISelectOption<unknown, any>,
): React.ReactNode {
    return option.data;
}

export default Select;
