import React, {useCallback, useRef} from 'react';
import {useDispatch} from 'react-redux';

import {IWithClassName} from 'types/withClassName';

import {setSnackbarInfo} from 'reducers/common/snackbar/actions';
import {ISnackbarState} from 'reducers/common/snackbar/reducer';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {
    EShareStatus,
    useNativeShareAPI,
} from 'utilities/hooks/useNativeShareAPI';

import * as i18nBlock from 'i18n/common-share';

import {IPopupProps} from 'components/Popup/Popup';
import SharePopup from 'components/Share/compoenents/SharePopup/SharePopup';

export interface IShareProps extends IWithClassName {
    button: React.ReactElement;
    buttonWrapperClassName?: string;
    link: string;

    popupDirections?: IPopupProps['directions'];

    onClick?: () => void;
    onInputFocus?: () => void;
    onShareLinkClick?: (text?: string) => void;
    onCopyLinkClick?: () => void;

    page: ISnackbarState['page'];
}

const Share: React.FC<IShareProps> = ({
    className,
    button,
    buttonWrapperClassName,
    link,
    onClick,
    onInputFocus,
    onShareLinkClick,
    onCopyLinkClick,
    popupDirections,
    page,
}) => {
    const {
        value: popupIsOpen,
        toggle: changePopupIsOpened,
        setFalse: closePopup,
    } = useBoolean(false);
    const dispatch = useDispatch();
    const buttonRef = useRef<HTMLDivElement>(null);

    const shareViaNativeAPI = useNativeShareAPI();

    const handleClick = useCallback(async () => {
        if (!popupIsOpen) {
            onClick?.();
        }

        const nativeStatus = await shareViaNativeAPI({url: link});

        if (nativeStatus.status === EShareStatus.OK) {
            onCopyLinkClick?.();

            return;
        }

        if (nativeStatus.status !== EShareStatus.UNSUPPORTED) {
            return;
        }

        changePopupIsOpened();
    }, [onClick, changePopupIsOpened, popupIsOpen, shareViaNativeAPI, link]);

    const handleCopyLink = () => {
        onCopyLinkClick?.();

        dispatch(
            setSnackbarInfo({
                lastAction: `copy-${link}`,
                hasCancellation: false,
                message: i18nBlock.linkCopied(),
                page,
            }),
        );
    };

    return (
        <div className={className}>
            <div
                onClick={handleClick}
                ref={buttonRef}
                className={buttonWrapperClassName}
            >
                {button}
            </div>

            <SharePopup
                isVisible={popupIsOpen}
                onClose={closePopup}
                link={link}
                onCopyLinkClick={handleCopyLink}
                onShareLinkClick={onShareLinkClick}
                onInputFocus={onInputFocus}
                popupDirections={popupDirections}
                popupAnchor={buttonRef}
            />
        </div>
    );
};

export default Share;
