import React, {useCallback, useRef} from 'react';

import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {deviceMods} from 'utilities/stylesUtils';
import {socialNetsInDefaultOrder} from 'components/ShareLink/utilities/socialNets';
import getSocialSharingNetText from 'components/ShareDropout/components/ShareDropoutPopup/utilities/getSocialSharingNetText';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nFavoritesBlock from 'i18n/account-Favorites';

import Box from 'components/Box/Box';
import Input from 'components/Input/Input';
import Button from 'components/Button/Button';
import Separator from 'components/Separator/Separator';
import {ShareLink} from 'components/ShareLink/ShareLink';
import PopupOrBottomSheet from 'components/PopupOrBottomSheet/PopupOrBottomSheet';
import {IPopupProps} from 'components/Popup/Popup';

import cx from './SharePopup.scss';

export interface ISharePopupProps {
    isVisible: boolean;
    link: string;

    popupDirections?: IPopupProps['directions'];
    popupAnchor: IPopupProps['anchor'];

    onClose: () => void;
    onInputFocus?: () => void;
    onShareLinkClick?: (text?: string) => void;
    onCopyLinkClick?: () => void;
}

const DEFAULT_DIRECTIONS = [EPopupDirection.BOTTOM_RIGHT];

const SharePopup: React.FC<ISharePopupProps> = ({
    isVisible,
    link,
    onClose,
    onInputFocus,
    onCopyLinkClick,
    onShareLinkClick,
    popupDirections = DEFAULT_DIRECTIONS,
    popupAnchor,
}) => {
    const inputRef = useRef<Input>(null);
    const deviceType = useDeviceType();

    const {isMobile} = deviceType;

    const handleCopyLink = useCallback((): void => {
        if (inputRef.current) {
            inputRef.current.select();
            document.execCommand('copy');
        }

        onCopyLinkClick?.();
    }, [onCopyLinkClick]);

    const handleFocus = useCallback(() => {
        handleCopyLink();
        onInputFocus?.();
    }, [onInputFocus, handleCopyLink]);

    return (
        <PopupOrBottomSheet
            isVisible={isVisible}
            onClose={onClose}
            popupProps={{
                anchor: popupAnchor,
                directions: popupDirections,
            }}
        >
            <Box className={cx('content', deviceMods('content', deviceType))}>
                <div className={cx('inputWrapper')}>
                    <Input
                        className={cx('input')}
                        value={link}
                        ref={inputRef}
                        size={isMobile ? 'l' : 'm'}
                        onFocus={handleFocus}
                    />

                    <Button
                        size={isMobile ? 'l' : 'm'}
                        onClick={handleCopyLink}
                        theme="primary"
                        width={isMobile ? 'max' : 'auto'}
                        className={cx('button')}
                    >
                        {i18nFavoritesBlock.copyLink()}
                    </Button>
                </div>

                <Separator margin={1} />

                <div className={cx('listVertical')}>
                    {socialNetsInDefaultOrder.map(type => (
                        <ShareLink
                            key={type}
                            type={type}
                            url={link}
                            directionStyle="vertical"
                            textAfter={getSocialSharingNetText(type)}
                            buttonClassName={cx('shareLink')}
                            onClick={onShareLinkClick}
                        />
                    ))}
                </div>
            </Box>
        </PopupOrBottomSheet>
    );
};

export default SharePopup;
