import React from 'react';

import {IInputChangeEvent} from 'components/Input/types/InputProps';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {socialNetsInDefaultOrder} from 'components/ShareLink/utilities/socialNets';

import * as i18nBlock from 'i18n/common-share';

import {ShareLink} from 'components/ShareLink/ShareLink';
import Button from 'components/Button/Button';
import Input from 'components/Input/Input';
import {
    withUpdateTinyUrl,
    IUpdateLink,
} from 'components/withUpdateTinyUrl/withUpdateTinyUrl';

import cx from './ShareHorizontal.scss';

interface IShareHorizontalState {
    inputValue: string;
}

class _ShareHorizontal extends React.PureComponent<
    IUpdateLink & IWithQaAttributes,
    IShareHorizontalState
> {
    private readonly inputRef = React.createRef<Input>();

    readonly state: IShareHorizontalState = {
        inputValue: this.props.tinyUrl,
    };

    componentDidUpdate(prevProps: IUpdateLink): void {
        if (prevProps.tinyUrl !== this.props.tinyUrl) {
            this.setState({inputValue: this.props.tinyUrl});
        }
    }

    private renderSocialLinks(): React.ReactNode {
        const qa = prepareQaAttributes({
            parent: this.props,
            current: 'socialLinks',
        });

        return (
            <div className={cx('listHorizontal')}>
                {socialNetsInDefaultOrder.map(type => (
                    <ShareLink
                        key={type}
                        type={type}
                        url={this.props.tinyUrl}
                        directionStyle="horizontal"
                        {...prepareQaAttributes({
                            parent: qa,
                            current: 'link',
                            key: type,
                        })}
                    />
                ))}
            </div>
        );
    }

    private copyLink = (): void => {
        if (!this.inputRef.current) {
            return;
        }

        this.inputRef.current.select();

        try {
            document.execCommand('copy');
        } catch (e) {
            // ignore error
        }
    };

    private onChange = (event: IInputChangeEvent, inputValue: string): void =>
        this.setState({inputValue});

    private onFocus = (): null | void =>
        this.inputRef.current && this.inputRef.current.select();

    render(): React.ReactNode {
        return (
            <div className={cx('root')}>
                <Input
                    className={cx('input-for-url')}
                    value={this.state.inputValue}
                    ref={this.inputRef}
                    onChange={this.onChange}
                    onFocus={this.onFocus}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'link',
                    })}
                />
                <Button
                    size="m-inset"
                    className={cx('copy-button')}
                    onClick={this.copyLink}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'copy',
                    })}
                >
                    {i18nBlock.copy()}
                </Button>
                {this.renderSocialLinks()}
            </div>
        );
    }
}

export const ShareHorizontal = withUpdateTinyUrl(_ShareHorizontal);
