import React, {useCallback} from 'react';

import {IWithClassName} from 'types/withClassName';
import {ESortType} from 'types/common/sort/ESortType';
import {ESortDirection} from 'types/common/sort/ESortDirection';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getSortTypeLabel from 'utilities/getSortTypeLabel';

import SortAscendingIcon from 'icons/12/SortAscending';
import SortDescendingIcon from 'icons/12/SortDescending';
import LinkButton from 'components/LinkButton/LinkButton';

import cx from './SortBy.scss';

interface ISortByProps extends IWithClassName, IWithQaAttributes {
    type: ESortType;
    direction: ESortDirection;
    isActive: boolean;
    onClick(type: ESortType, direction: ESortDirection): void;
}

const SortBy: React.FC<ISortByProps> = props => {
    const {className, type, direction, isActive, onClick} = props;

    const handleClick = useCallback(() => {
        onClick(
            type,
            isActive && direction === ESortDirection.ASC
                ? ESortDirection.DESC
                : ESortDirection.ASC,
        );
    }, [type, direction, isActive, onClick]);

    const icon =
        direction === ESortDirection.ASC ? (
            <SortAscendingIcon className={cx('icon')} />
        ) : (
            <SortDescendingIcon className={cx('icon')} />
        );

    return (
        <LinkButton
            className={cx('root', className)}
            theme={isActive ? 'black' : 'normal'}
            onClick={handleClick}
            {...prepareQaAttributes(props)}
        >
            {getSortTypeLabel(type)}

            {isActive && icon}
        </LinkButton>
    );
};

export default SortBy;
