import React, {useCallback, useMemo} from 'react';

import {ESortType} from 'types/common/sort/ESortType';
import {ESortDirection} from 'types/common/sort/ESortDirection';
import {IWithClassName} from 'types/withClassName';

import getSortTypeLabel from 'utilities/getSortTypeLabel';
import getSortDirectionLabel from 'utilities/getSortDirectionLabel';
import capitalizeFirstLetter from 'utilities/strings/capitalizeFirstLetter';

import * as i18nCommonSortBlock from 'i18n/common-sort';

import Select from 'components/Select/Select';
import Box, {IBoxProps} from 'components/Box/Box';
import Flex from 'components/Flex/Flex';

import cx from './SortMobile.scss';

interface ISortMobileProps
    extends IWithClassName,
        Partial<Pick<IBoxProps, 'above' | 'below'>> {
    sortOrder?: ESortType[];
    type: ESortType;
    direction: ESortDirection;
    onChange(sortType: ESortType, sortDirection: ESortDirection): void;
}

export const SORTS_ORDER = [
    ESortType.DEPARTURE,
    ESortType.ARRIVAL,
    ESortType.PRICE,
    ESortType.DURATION,
];

const SortMobile: React.FC<ISortMobileProps> = props => {
    const {
        className,
        sortOrder = SORTS_ORDER,
        type,
        direction,
        onChange,
        above,
        below,
    } = props;

    const sortTypeOptions = useMemo(() => {
        return sortOrder.map(value => ({
            value,
            data: capitalizeFirstLetter(getSortTypeLabel(value)),
        }));
    }, [sortOrder]);
    const sortDirectionOptions = useMemo(() => {
        return Object.values(ESortDirection).map(direction => ({
            value: direction,
            data: getSortDirectionLabel(type, direction),
        }));
    }, [type]);

    const handleChangeSortType = useCallback(
        (t: ESortType) => {
            onChange(t, direction);
        },
        [direction, onChange],
    );
    const handleChangeSortDirection = useCallback(
        (d: ESortDirection) => {
            onChange(type, d);
        },
        [type, onChange],
    );

    return (
        <Box className={className} between={4} above={above} below={below}>
            <Box textSize="m" textWeight="bold">
                {i18nCommonSortBlock.title()}
            </Box>

            <Flex between={3} flexDirection={'column'}>
                <div className={cx('selectWrapper')}>
                    <Select
                        value={type}
                        options={sortTypeOptions}
                        onChange={handleChangeSortType}
                        width="max"
                        size="l"
                    />
                </div>
                <div className={cx('selectWrapper')}>
                    <Select
                        value={direction}
                        options={sortDirectionOptions}
                        onChange={handleChangeSortDirection}
                        width="max"
                        size="l"
                    />
                </div>
            </Flex>
        </Box>
    );
};

export default SortMobile;
