import React, {useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IPromoCodeForSubscription} from 'server/api/NotifierApi/types';
import {ESubscriptionPromoCodeType} from 'types/subscription/ESubscriptionPromoCodeType';
import {TPictureSize} from '../../types/TPictureSize';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {CurrencyType} from 'utilities/currency/CurrencyType';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {
    PICTURE_SIZES,
    getPromoCodePictureSource,
} from './utilities/pictureSource';

import * as i18nBlock from 'i18n/promoCodeForSubscription';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Link from 'components/Link/Link';
import Price from 'components/Price/Price';
import Heading from 'components/Heading/Heading';
import ImageIcon from 'components/ImageIcon/ImageIcon';
import SubscriptionForm, {
    ISubscriptionFormProps,
} from 'components/SubscriptionForm/SubscriptionForm';

import cx from './SubscriptionWithPromoCode.scss';

export interface IPromoCodeForSubscriptionContentProps
    extends IWithClassName,
        Omit<ISubscriptionFormProps, 'theme'> {
    promoCode: IPromoCodeForSubscription;
    pictureSize?: TPictureSize;
    pictureClassName?: string;
    /** Настройка для десктопного отображения на поиске / на главной */
    headerSize?: 'xl' | 'xxl';
}

const PromoCodeForSubscriptionContent: React.FC<IPromoCodeForSubscriptionContentProps> =
    props => {
        const {
            promoCode,
            pictureSize = 'm',
            pictureClassName,
            className,
            headerSize,
            ...rest
        } = props;
        const {amount, type, minTotalCost} = promoCode;
        const deviceType = useDeviceType();
        const {isDesktop} = deviceType;

        const title = useMemo(() => {
            return type === ESubscriptionPromoCodeType.FIX
                ? insertJSXIntoKey(i18nBlock.discountFix)({
                      amount: (
                          <Price currency={CurrencyType.RUB} value={amount} />
                      ),
                  })
                : i18nBlock.discountPercent({amount, isDesktop});
        }, [amount, type, isDesktop]);

        const description = useMemo(() => {
            if (isDesktop) {
                if (minTotalCost) {
                    return insertJSXIntoKey(i18nBlock.descriptionMinTotalCost)({
                        minTotalCost: (
                            <Price
                                currency={CurrencyType.RUB}
                                value={minTotalCost}
                            />
                        ),
                    });
                }

                return i18nBlock.description();
            }

            if (minTotalCost) {
                return insertJSXIntoKey(i18nBlock.descriptionShortMinTotalCost)(
                    {
                        minTotalCost: (
                            <Price
                                currency={CurrencyType.RUB}
                                value={minTotalCost}
                            />
                        ),
                    },
                );
            }

            return i18nBlock.descriptionShort();
        }, [minTotalCost, isDesktop]);

        return (
            <div
                className={cx(
                    'root',
                    deviceMods('root', deviceType),
                    className,
                )}
            >
                <ImageIcon
                    className={cx('picture', pictureClassName)}
                    src={getPromoCodePictureSource(promoCode, pictureSize)}
                    {...PICTURE_SIZES[pictureSize]}
                />

                <Box className={cx('content')} between="5">
                    <Flex
                        className={cx(
                            'header',
                            headerSize && `header_size_${headerSize}`,
                        )}
                        between="1"
                        flexDirection="column"
                    >
                        <Heading level={2}>{title}</Heading>
                        <Text size="m">{description}</Text>
                    </Flex>

                    <Flex
                        className={cx('form')}
                        between="2"
                        flexDirection="column"
                    >
                        <SubscriptionForm
                            size="m-inset"
                            inputClassName={cx('input')}
                            inputPlaceholder={i18nBlock.emailPlaceholder()}
                            showErrorsInTooltip
                            {...rest}
                        />

                        <Text className={cx('disclaimer')} size="s">
                            {insertJSXIntoKey(i18nBlock.acceptTerms)({
                                link: (
                                    <Link
                                        url={i18nBlock.userAgreementLink()}
                                        theme="normal"
                                        target="_blank"
                                    >
                                        {i18nBlock.userAgreement()}
                                    </Link>
                                ),
                            })}
                        </Text>
                    </Flex>
                </Box>
            </div>
        );
    };

export default PromoCodeForSubscriptionContent;
