import React, {
    ReactNode,
    useCallback,
    useMemo,
    useState,
    FocusEvent,
} from 'react';
import {FormApi, Mutator} from 'final-form';
import {FormRenderProps} from 'react-final-form';
import setFieldData from 'final-form-set-field-data';

import {EFormKey} from 'constants/form/EFormKey';
import validationInfo from './constants/validationInfo';

import {
    EFieldName,
    EGroupName,
    ISubscriptionModalFormValues,
} from './types/form';
import {IWithClassName} from 'types/withClassName';
import {AppError} from 'types/common/AppError';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {delay} from 'utilities/async/delay';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nBlock from 'i18n/components-SubscriptionModal';

import Button from 'components/Button/Button';
import FormField from 'components/FormField/FormField';
import setFormErrors, {
    TSetFormErrors,
} from 'components/Form/mutators/setFormErrors';
import focusFirstInvalidField, {
    TFocusFirstInvalidField,
} from 'components/Form/mutators/focusFirstInvalidField';
import Form from 'components/Form/Form';
import createSubmitErrorDecorator from 'components/Form/decorators/createSubmitErrorDecorator';
import InputWithLoader, {
    EInputLoadingStatus,
} from 'components/InputWithLoader/InputWithLoader';

import cx from './SubscriptionForm.scss';

const focusOnSubmitErrors =
    createSubmitErrorDecorator<ISubscriptionModalFormValues>(
        EFormKey.MODAL_SUBSCRIPTION,
    );

export interface ISubscriptionFormProps extends IWithClassName {
    submitButton?: React.ReactNode;
    inputPlaceholder?: string;
    email?: string;
    inputClassName?: string;
    buttonClassName?: string;
    size?: 'l' | 'm' | 'm-inset';
    restoreDefaultStateTimeout?: number;
    showErrorsInTooltip?: boolean;

    onSubmit(email: string): void | Promise<void>;
    onBeforeSubmit?(): void;
}

const setEmail: Mutator<
    ISubscriptionModalFormValues,
    ISubscriptionModalFormValues
> = ([value], state, {changeValue}): void => {
    changeValue(
        state,
        `${EGroupName.contacts}.${EFieldName.email}`,
        () => value,
    );
};

const SubscriptionForm: React.FC<ISubscriptionFormProps> = props => {
    const {
        email = '',
        submitButton = i18nBlock.button(),
        inputPlaceholder = i18nBlock.inputPlaceholder(),
        onSubmit,
        onBeforeSubmit,
        className,
        inputClassName,
        buttonClassName,
        size = 'l',
        restoreDefaultStateTimeout,
        showErrorsInTooltip,
    } = props;

    const deviceType = useDeviceType();
    const [loading, setLoading] = useState<EInputLoadingStatus>(
        EInputLoadingStatus.DEFAULT,
    );

    const initialValues = useMemo(
        () => ({
            [EGroupName.contacts]: {
                [EFieldName.email]: email,
            },
        }),
        [email],
    );

    const handleSubscribe = useCallback(
        async (
            formValues: ISubscriptionModalFormValues,
            form: FormApi<ISubscriptionModalFormValues>,
        ) => {
            try {
                const emailValue =
                    formValues[EGroupName.contacts]?.[EFieldName.email];

                if (!emailValue) {
                    return i18nBlock.emailRequired();
                }

                setLoading(EInputLoadingStatus.LOADING);

                await onSubmit(emailValue.trim());

                setLoading(EInputLoadingStatus.DONE);

                if (restoreDefaultStateTimeout) {
                    await delay(restoreDefaultStateTimeout);

                    // Возвращаем исходное состояние
                    setLoading(EInputLoadingStatus.DEFAULT);
                }
            } catch (e) {
                const errorMessage =
                    e instanceof AppError ? e.message : i18nBlock.sendError();
                const errors = {
                    [EGroupName.contacts]: {
                        [EFieldName.email]: errorMessage,
                    },
                };

                (form.mutators.setFormErrors as TSetFormErrors)(errors);
                (
                    form.mutators
                        .focusFirstInvalidField as TFocusFirstInvalidField<ISubscriptionModalFormValues>
                )(EFormKey.MODAL_SUBSCRIPTION, errors);

                setLoading(EInputLoadingStatus.DEFAULT);
            }
        },
        [onSubmit, setLoading, restoreDefaultStateTimeout],
    );

    const handleBeforeSubmit = (
        callback: FormRenderProps['handleSubmit'],
        event: React.FormEvent<HTMLFormElement>,
    ): void => {
        onBeforeSubmit?.();
        callback(event);
    };

    const handleInputFocus = useCallback(
        (
            event: FocusEvent<HTMLInputElement>,
            form: FormApi<ISubscriptionModalFormValues>,
            input: {onFocus: (e: FocusEvent<HTMLInputElement>) => void},
        ) => {
            if (loading === EInputLoadingStatus.DONE) {
                setLoading(EInputLoadingStatus.DEFAULT);
                form.mutators.setEmail('');
            }

            input.onFocus?.(event);
        },
        [loading, setLoading],
    );

    return (
        <Form<ISubscriptionModalFormValues>
            onSubmit={handleSubscribe}
            decorators={[focusOnSubmitErrors]}
            mutators={{
                focusFirstInvalidField,
                setFieldData,
                setFormErrors,
                setEmail,
            }}
            initialValues={initialValues}
            validationInfo={validationInfo}
            subscription={{}}
            render={({handleSubmit, form}): ReactNode => (
                <form
                    className={cx(
                        'form',
                        deviceMods('form', deviceType),
                        className,
                    )}
                    name={EFormKey.MODAL_SUBSCRIPTION}
                    onSubmit={
                        onBeforeSubmit
                            ? handleBeforeSubmit.bind(null, handleSubmit)
                            : handleSubmit
                    }
                    autoComplete="off"
                >
                    <Form.FieldGroup groupId={EGroupName.contacts}>
                        <FormField
                            className={cx('inputWrap', inputClassName)}
                            name={EFieldName.email}
                            deviceType={deviceType}
                            showErrorsInTooltip={showErrorsInTooltip}
                            control={(
                                {input},
                                {error, inputRef, controlRef},
                            ): ReactNode => (
                                <InputWithLoader
                                    {...input}
                                    id={input.name}
                                    inputRef={(currentInputRef): void => {
                                        inputRef.current = currentInputRef;
                                    }}
                                    innerRef={controlRef}
                                    value={input.value}
                                    size={size}
                                    state={error ? 'error' : undefined}
                                    type="text"
                                    inputMode="email"
                                    placeholder={inputPlaceholder}
                                    loading={loading}
                                    autoComplete={error ? 'off' : undefined}
                                    onFocus={(
                                        e: FocusEvent<HTMLInputElement>,
                                    ): void => handleInputFocus(e, form, input)}
                                />
                            )}
                        />
                        <Button
                            className={buttonClassName}
                            size={size}
                            width={deviceType.isMobile ? 'max' : 'auto'}
                            theme="primary"
                            type="submit"
                            disabled={loading !== EInputLoadingStatus.DEFAULT}
                        >
                            {submitButton}
                        </Button>
                    </Form.FieldGroup>
                </form>
            )}
        />
    );
};

export default SubscriptionForm;
