import React, {useMemo, useRef} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IPromoCodeForSubscription} from 'server/api/NotifierApi/types';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import getTitle from './utilities/getTitle';
import getDescription from './utilities/getDescription';

import * as i18n from 'i18n/promoCodeForSubscription';

import Card from 'components/Card/Card';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import SubscriptionForm, {
    ISubscriptionFormProps,
} from 'components/SubscriptionForm/SubscriptionForm';
import Link from 'components/Link/Link';
import TextSkeleton from 'components/Skeletons/TextSkeleton/TextSkeleton';
import Mail3DImage from 'icons/raster-illustrations/L/Mail3D';

import ScopeContext from 'contexts/ScopeContext';

import cx from './SubscriptionInlineNew.scss';

interface ISubscriptionInlineNewProps
    extends IWithClassName,
        Pick<ISubscriptionFormProps, 'onBeforeSubmit' | 'onSubmit' | 'email'> {
    promoCode?: IPromoCodeForSubscription;
    loading?: boolean;
}

const SubscriptionInlineNew: React.FC<ISubscriptionInlineNewProps> = props => {
    const {
        className,
        promoCode,
        loading = false,
        onBeforeSubmit,
        onSubmit,
        email,
    } = props;

    const deviceType = useDeviceType();

    const cardRef = useRef<HTMLElement>(null);

    const title = useMemo(
        () => getTitle({promoCode, deviceType}),
        [deviceType, promoCode],
    );
    const description = useMemo(
        () => getDescription({promoCode, deviceType}),
        [deviceType, promoCode],
    );

    return (
        <Card
            className={cx(className, 'root', deviceMods('root', deviceType))}
            background="grey"
            ref={cardRef}
            x={deviceType.isDesktop ? 8 : 4}
            y={deviceType.isDesktop ? undefined : 4}
        >
            <ScopeContext.Provider value={cardRef.current || undefined}>
                <Flex
                    className={cx('content')}
                    flexDirection={
                        deviceType.isDesktop ? 'row' : 'column-reverse'
                    }
                    justifyContent="space-between"
                    alignItems="center"
                    inline={deviceType.isDesktop}
                    between={deviceType.isDesktop ? 4 : undefined}
                    nowrap
                >
                    <Flex
                        flexDirection="column"
                        between={deviceType.isDesktop ? 4 : 6}
                        y={deviceType.isDesktop ? 8 : undefined}
                    >
                        <Flex
                            flexDirection="column"
                            between={deviceType.isDesktop ? 1 : 3}
                        >
                            {loading ? (
                                <TextSkeleton
                                    className={cx('titleSkeleton')}
                                    size={deviceType.isDesktop ? 'xxl' : 'xl'}
                                    withAnimation
                                />
                            ) : (
                                <Text
                                    size={deviceType.isDesktop ? 'xxl' : 'xl'}
                                    weight="fat"
                                    align={
                                        deviceType.isDesktop
                                            ? undefined
                                            : 'center'
                                    }
                                >
                                    {title}
                                </Text>
                            )}
                            {loading ? (
                                <>
                                    <TextSkeleton
                                        className={cx('descriptionSkeleton')}
                                        size="m"
                                        withAnimation
                                    />
                                    {deviceType.isMobile && (
                                        <TextSkeleton
                                            className={cx(
                                                'descriptionSkeleton',
                                            )}
                                            size="m"
                                            withAnimation
                                        />
                                    )}
                                </>
                            ) : (
                                <Text
                                    align={
                                        deviceType.isDesktop
                                            ? undefined
                                            : 'center'
                                    }
                                >
                                    {description}
                                </Text>
                            )}
                        </Flex>
                        <Flex
                            flexDirection="column"
                            between={3}
                            below={deviceType.isDesktop ? undefined : 4}
                        >
                            <SubscriptionForm
                                inputClassName={cx('input')}
                                buttonClassName={cx('button')}
                                email={email}
                                onBeforeSubmit={onBeforeSubmit}
                                onSubmit={onSubmit}
                            />
                            <Text
                                color="secondary"
                                size="s"
                                align={
                                    deviceType.isDesktop ? undefined : 'center'
                                }
                            >
                                {insertJSXIntoKey(i18n.acceptTerms)({
                                    link: (
                                        <Link
                                            url={i18n.userAgreementLink()}
                                            theme="ghost"
                                            target="_blank"
                                        >
                                            {i18n.userAgreement()}
                                        </Link>
                                    ),
                                })}
                            </Text>
                        </Flex>
                    </Flex>

                    <Mail3DImage
                        className={cx('image')}
                        alt={i18n.imageAlt()}
                    />
                </Flex>
            </ScopeContext.Provider>
        </Card>
    );
};

export default SubscriptionInlineNew;
