import {Meta} from '@storybook/react/types-6-0';
import React, {useState} from 'react';
import times from 'lodash/times';
import routerDecorator from 'storybook/storyDecorators/routerDecorator';
import {useHistory} from 'react-router-dom';

import {TStorybookFunctionComponent} from 'storybook/types';

import Tabs from 'components/Tabs/Tabs';
import SupportPhone from 'components/SupportPhone/SupportPhone';
import Flex from 'components/Flex/Flex';
import Checkbox from 'components/Checkbox/Checkbox';

export default {
    title: 'Example/Tabs',
    component: Tabs,
    decorators: [routerDecorator],
} as Meta;

const tabs = times(5).map(i => ({
    id: `${i}`,
    name: `Tab item ${i}`,
}));

export const Simple: TStorybookFunctionComponent = () => {
    const [activeTabId, setActiveTabId] = useState(tabs[1].id);

    return (
        <Tabs
            tabs={tabs}
            activeTabId={activeTabId}
            onChange={(tab): void => {
                setActiveTabId(tab.id);
            }}
            getTabId={(tab): string => tab.id}
            renderTab={({tab, tabProps}): React.ReactNode => {
                return (
                    <Tabs.Tab key={tab.id} {...tabProps}>
                        {tab.name}
                    </Tabs.Tab>
                );
            }}
        />
    );
};

export const WithAdditional: TStorybookFunctionComponent = () => {
    const [activeTabId, setActiveTabId] = useState(tabs[1].id);

    return (
        <Tabs
            border
            additional={<SupportPhone />}
            tabs={tabs}
            activeTabId={activeTabId}
            onChange={(tab): void => {
                setActiveTabId(tab.id);
            }}
            getTabId={(tab): string => tab.id}
            renderTab={({tab, tabProps}): React.ReactNode => {
                return (
                    <Tabs.Tab key={tab.id} {...tabProps}>
                        {tab.name}
                    </Tabs.Tab>
                );
            }}
        />
    );
};

export const WithBorder: TStorybookFunctionComponent = () => {
    const [activeTabId, setActiveTabId] = useState(tabs[1].id);

    return (
        <Tabs
            border
            tabs={tabs}
            activeTabId={activeTabId}
            onChange={(tab): void => {
                setActiveTabId(tab.id);
            }}
            getTabId={(tab): string => tab.id}
            renderTab={({tab, tabProps}): React.ReactNode => {
                return (
                    <Tabs.Tab key={tab.id} {...tabProps}>
                        {tab.name}
                    </Tabs.Tab>
                );
            }}
        />
    );
};

export const WithArrow: TStorybookFunctionComponent = () => {
    const [activeTabId, setActiveTabId] = useState(tabs[1].id);

    return (
        <Tabs
            arrow
            tabs={tabs}
            activeTabId={activeTabId}
            onChange={(tab): void => {
                setActiveTabId(tab.id);
            }}
            getTabId={(tab): string => tab.id}
            renderTab={({tab, tabProps}): React.ReactNode => {
                return (
                    <Tabs.Tab key={tab.id} {...tabProps}>
                        {tab.name}
                    </Tabs.Tab>
                );
            }}
        />
    );
};

export const WithBorderAndArrow: TStorybookFunctionComponent = () => {
    const [activeTabId, setActiveTabId] = useState(tabs[1].id);

    return (
        <Tabs
            arrow
            border
            tabs={tabs}
            activeTabId={activeTabId}
            onChange={(tab): void => {
                setActiveTabId(tab.id);
            }}
            getTabId={(tab): string => tab.id}
            renderTab={({tab, tabProps}): React.ReactNode => {
                return (
                    <Tabs.Tab key={tab.id} {...tabProps}>
                        {tab.name}
                    </Tabs.Tab>
                );
            }}
        />
    );
};

export const Bold: TStorybookFunctionComponent = () => {
    const [activeTabId, setActiveTabId] = useState(tabs[1].id);

    return (
        <Tabs
            bold
            tabs={tabs}
            activeTabId={activeTabId}
            onChange={(tab): void => {
                setActiveTabId(tab.id);
            }}
            getTabId={(tab): string => tab.id}
            renderTab={({tab, tabProps}): React.ReactNode => {
                return (
                    <Tabs.Tab key={tab.id} {...tabProps}>
                        {tab.name}
                    </Tabs.Tab>
                );
            }}
        />
    );
};

export const TwoRowTab: TStorybookFunctionComponent = () => {
    const [activeTabId, setActiveTabId] = useState(tabs[1].id);

    return (
        <Tabs
            tabs={tabs}
            activeTabId={activeTabId}
            onChange={(tab): void => {
                setActiveTabId(tab.id);
            }}
            getTabId={(tab): string => tab.id}
            renderTab={({tab, tabProps}): React.ReactNode => {
                return (
                    <Tabs.Tab key={tab.id} {...tabProps}>
                        <div>
                            <div>{tab.name}</div>
                            <div
                                style={{
                                    fontSize: 13,
                                    lineHeight: '16px',
                                    marginTop: 2,
                                }}
                            >
                                Вагон №2
                            </div>
                        </div>
                    </Tabs.Tab>
                );
            }}
        />
    );
};

export const AutoScrollToActiveTab: TStorybookFunctionComponent = () => {
    const [activeTabId, setActiveTabId] = useState(tabs[3].id);

    return (
        <div style={{maxWidth: 320}}>
            <Tabs
                border
                tabs={tabs}
                activeTabId={activeTabId}
                onChange={(tab): void => {
                    setActiveTabId(tab.id);
                }}
                getTabId={(tab): string => tab.id}
                renderTab={({tab, tabProps}): React.ReactNode => {
                    return (
                        <Tabs.Tab key={tab.id} {...tabProps}>
                            {tab.name}
                        </Tabs.Tab>
                    );
                }}
            />
        </div>
    );
};

export const Link: TStorybookFunctionComponent = () => {
    const history = useHistory();
    const activeTabId = history.location.hash.slice(1);

    return (
        <Tabs
            border
            tabs={tabs}
            activeTabId={activeTabId}
            getTabId={(tab): string => tab.id}
            renderTab={({tab, tabProps}): React.ReactNode => {
                return (
                    <Tabs.Tab key={tab.id} {...tabProps} to={`#${tab.id}`}>
                        {tab.name}
                    </Tabs.Tab>
                );
            }}
        />
    );
};

export const WithDisabled: TStorybookFunctionComponent = () => {
    const [activeTabId, setActiveTabId] = useState(tabs[1].id);
    const [disabledTabs, setDisabledTabs] = useState<
        Partial<Record<string, boolean>>
    >({'0': true});
    const [arrow, setArrow] = useState(true);
    const [border, setBorder] = useState(true);
    const [bold, setBold] = useState(false);

    return (
        <Flex flexDirection="column" between={10}>
            <Tabs
                arrow={arrow}
                border={border}
                bold={bold}
                tabs={tabs}
                activeTabId={activeTabId}
                onChange={(tab): void => {
                    setActiveTabId(tab.id);
                }}
                getTabId={(tab): string => tab.id}
                getTabIsDisabled={(tab): boolean =>
                    Boolean(disabledTabs[tab.id])
                }
                renderTab={({tab, tabProps}): React.ReactNode => {
                    return (
                        <Tabs.Tab key={tab.id} {...tabProps}>
                            {tab.name}
                        </Tabs.Tab>
                    );
                }}
            />

            <Flex flexDirection="column" between={2}>
                {tabs.map(tab => (
                    <Checkbox
                        key={tab.id}
                        checked={disabledTabs[tab.id]}
                        label={`Заблокировать таб № ${tab.id}`}
                        disabled={tab.id === activeTabId}
                        onChange={(e): void => {
                            setDisabledTabs(prevState => ({
                                ...prevState,
                                [tab.id]: e.target.checked,
                            }));
                        }}
                    />
                ))}
            </Flex>

            <Flex flexDirection="column" between={2}>
                <Checkbox
                    checked={arrow}
                    label="Показать разделитель между табами"
                    onChange={(e): void => {
                        setArrow(e.target.checked);
                    }}
                />
                <Checkbox
                    checked={border}
                    label="Показать нижнюю границу"
                    onChange={(e): void => {
                        setBorder(e.target.checked);
                    }}
                />
                <Checkbox
                    checked={bold}
                    label="Вариант Bold"
                    onChange={(e): void => {
                        setBold(e.target.checked);
                    }}
                />
            </Flex>
        </Flex>
    );
};
