import React from 'react';
import {LocationDescriptor} from 'history';

import {IWithClassName} from 'types/withClassName';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareCustomQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Link from 'components/Link/Link';

import cx from './Tab.scss';

export interface ITabProps
    extends IWithClassName,
        IWithQaAttributes,
        React.HTMLAttributes<HTMLButtonElement> {
    /**
     * Помечает этот элемент активным, устанавливая красный border снизу
     */
    active: boolean;

    /**
     * Большой вариант таба
     */
    bold: boolean;

    disabled?: boolean;

    onClick?: React.MouseEventHandler;

    innerRef: React.Ref<HTMLButtonElement> | undefined;

    url?: string;
    to?: LocationDescriptor;
}

const Tab: React.FC<ITabProps> = props => {
    const {
        className,
        active,
        bold,
        children,
        innerRef,
        url,
        to,
        disabled = false,
        onClick,
        ...rest
    } = props;

    const deviceType = useDeviceType();

    const hasLink = (url || to) && !active && !disabled;

    const content = (
        <button
            className={cx(
                'button',
                deviceMods('button', deviceType),
                {
                    button_active: active,
                    button_bold: bold,
                    button_withoutPointer: !onClick && !url && !to,
                    root_disabled: disabled,
                },
                !hasLink && className,
            )}
            ref={innerRef}
            onClick={disabled ? undefined : onClick}
            disabled={disabled}
            type="button"
            {...prepareCustomQaAttributes('data-active', String(active))}
            {...prepareQaAttributes(rest)}
            {...rest}
        >
            {children}
        </button>
    );

    if (!hasLink) {
        return content;
    }

    const linkClassName = cx('link', className);

    if (url) {
        return (
            <Link className={linkClassName} url={url}>
                {content}
            </Link>
        );
    }

    if (to) {
        return (
            <Link className={linkClassName} to={to}>
                {content}
            </Link>
        );
    }

    return content;
};

export default Tab;
