import React from 'react';

import {IWithClassName} from 'types/withClassName';

import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import cx from './Text.scss';

export const textSizeCn = (size: TTextSize): string => cx(`size_${size}`);

export const textWeightCn = (weight: TTextWeight): string =>
    cx(`weight_${weight}`);

export const textColorCn = (color: TTextColor): string => cx(`color_${color}`);

const alignCn = (align: TTextAlign | undefined): string | undefined =>
    align ? cx(`align_${align}`) : undefined;

const overflowCn = (overflow: TOverflow | undefined): string | undefined =>
    overflow ? cx(`overflow_${overflow}`) : undefined;

export const whiteSpaceCn = (
    whiteSpace: TWhiteSpace | undefined,
): string | undefined =>
    whiteSpace ? cx(`whiteSpace_${whiteSpace}`) : undefined;

export interface ITextProps extends IWithQaAttributes, IWithClassName {
    /**
     * Вес текста
     *
     * @default 'normal'
     */
    weight?: TTextWeight;
    /**
     * Размер текста
     *
     * @default 'm'
     */
    size?: TTextSize;
    /**
     * Цвет текста
     *
     * @default 'primary'
     */
    color?: TTextColor;
    disabled?: boolean;
    whiteSpace?: TWhiteSpace;
    align?: TTextAlign;
    onClick?: () => void;
    tag?: keyof JSX.IntrinsicElements;
    tagProps?: React.HTMLAttributes<any> & React.PropsWithRef<any>;
    overflow?: TOverflow;
}

const Text: React.FC<ITextProps> = ({
    className,
    weight = 'normal',
    size = 'm',
    color = 'primary',
    disabled,
    whiteSpace,
    align,
    onClick,
    children,
    tag,
    tagProps,
    overflow,
    ...rest
}) => {
    const Tag = tag || 'span';

    return (
        <Tag
            {...prepareQaAttributes(rest)}
            onClick={onClick}
            {...tagProps}
            className={cx(
                className,
                textSizeCn(size),
                textWeightCn(weight),
                textColorCn(color),
                whiteSpaceCn(whiteSpace),
                alignCn(align),
                overflowCn(overflow),
                disabled && 'disabled',
            )}
        >
            {children}
        </Tag>
    );
};

export default Text;

export type TTextColor =
    | 'current'
    | 'primary'
    | 'secondary'
    | 'inverse'
    | 'link'
    | 'link-hover'
    | 'success'
    | 'highlight'
    | 'alert'
    | 'link-pale'
    | 'plus'
    | 'plus-cropped'
    | 'disabled';

export type TTextSize =
    | 'xs'
    | 's'
    | 's-inset'
    | 'm'
    | 'l'
    | 'xl'
    | 'xxl'
    | 'inherit';

export type TTextWeight = 'normal' | 'medium' | 'bold' | 'fat';

type TWhiteSpace = 'normal' | 'nowrap' | 'pre-wrap';

type TTextAlign = 'left' | 'right' | 'center' | 'justify';

type TOverflow = 'ellipsis';
