import React, {useCallback} from 'react';

import {LABEL_MAP, PERIOD_MAP} from 'components/TimeFilter/constants';

import {ETimeOfDay} from 'utilities/dateUtils/types';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import FilterCheckList from 'components/FilterCheckList/FilterCheckList';
import Box from 'components/Box/Box';

import cx from './TimeFilter.scss';

interface ITimeFilterProps
    extends IWithClassName,
        IWithQaAttributes,
        IWithDeviceType {
    title?: string;
    value: ETimeOfDay[];
    availableValues: ETimeOfDay[];
    options: ETimeOfDay[];
    renderPrice?: (option: ETimeOfDay, index?: number) => React.ReactNode;
    labelMap?: Record<ETimeOfDay, string>;
    periodMap?: Record<ETimeOfDay, string>;
    onChange(value: ETimeOfDay, checked: boolean): void;
}

const TimeFilter: React.FC<ITimeFilterProps> = props => {
    const {
        className,
        title,
        value,
        options,
        availableValues,
        renderPrice,
        labelMap = LABEL_MAP,
        periodMap = PERIOD_MAP,
        deviceType,
        onChange,
    } = props;

    const renderText = useCallback(
        (option: ETimeOfDay): React.ReactNode => {
            return <div className={cx('label')}>{labelMap[option]}</div>;
        },
        [labelMap],
    );
    const renderPeriod = useCallback(
        (option: ETimeOfDay): React.ReactNode => {
            return periodMap[option];
        },
        [periodMap],
    );

    if (options.length === 1) {
        return null;
    }

    return (
        <Box className={className} between={3} {...prepareQaAttributes(props)}>
            {title && (
                <Box
                    textSize="m"
                    textWeight="bold"
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'title',
                    })}
                >
                    {title}
                </Box>
            )}

            <FilterCheckList
                deviceType={deviceType}
                values={value}
                availableValues={availableValues}
                options={options}
                renderText={renderText}
                renderComment={renderPeriod}
                renderAside={renderPrice}
                onChange={onChange}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'checkList',
                })}
            />
        </Box>
    );
};

export default React.memo(TimeFilter);
