import React, {useCallback, useEffect} from 'react';
import {IntersectionOptions, useInView} from 'react-intersection-observer';

import {UIEvent} from 'types/events';
import {IWithClassName} from 'types/withClassName';
import {TImage} from 'types/common/TImage';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {
    ASYNC_LOADING_STATE,
    useAsyncState,
} from 'utilities/hooks/useAsyncState';
import {logError} from 'utilities/logger/logError';

import Spinner, {TSpinnerSize} from 'components/Spinner/Spinner';
import PictureIcon from 'icons/24/Picture';
import useSrcSet from 'components/TravelImage/hooks/useSrcSet';

import cx from './TravelImage.scss';

export interface IImageProps extends IWithClassName, IWithQaAttributes {
    src?: string;
    responsiveSet?: TImage[];
    id?: string | number;
    isCrop?: boolean;
    onClick?: (e: UIEvent, id?: string | number) => void;
    onLoad?: (image: HTMLImageElement) => void;
    onError?: (src?: string) => void;
    isPlaceholder?: boolean;
    isWide?: boolean;
    /** Кастомная заглушка (если картинки нет) */
    imageStub?: React.ReactNode;
    loaderClassName?: string;
    errorClassName?: string;
    imageClassName?: string;
    withoutImageClassName?: string;
    imageAlt?: string;
    spinnerSize?: TSpinnerSize;
    style?: React.CSSProperties;
    imageStyle?: React.CSSProperties;
    hasFitCover?: boolean;
    attrs?: object;
    imageRef?: React.Ref<HTMLImageElement>;
    /**
     * Грузить картинку при появлении ее во вьюпорте
     */
    isLoadOnViewportIntersect?: boolean;
    isRenderImmediately?: boolean;
}

const defaultImageStub = <PictureIcon className={cx('errorIcon')} />;

const INTERSECTION_OBSERVER_OPTIONS: IntersectionOptions = {
    rootMargin: '300px 100px 300px 100px',
    triggerOnce: true,
};

const TravelImage: React.FC<IImageProps> = props => {
    const {
        id,
        src,
        responsiveSet,
        isCrop = false,
        onLoad,
        onError,
        isPlaceholder,
        isWide,
        onClick,
        imageAlt,
        className,
        hasFitCover,
        spinnerSize,
        imageStub,
        errorClassName,
        imageClassName,
        loaderClassName,
        withoutImageClassName,
        style,
        imageStyle,
        attrs,
        imageRef,
        isLoadOnViewportIntersect = false,
        isRenderImmediately,
    } = props;

    const {
        isLoading,
        isSuccess,
        isError,
        success: setLoadSuccess,
        error: setLoadError,
    } = useAsyncState(ASYNC_LOADING_STATE);

    const handleClick = useCallback(
        (e: UIEvent): void => {
            if (onClick) {
                onClick(e, id);
            }
        },
        [onClick, id],
    );

    const handleOnSuccessLoad = useCallback(
        ({target}: Event): void => {
            setLoadSuccess();

            if (onLoad) {
                onLoad(target as HTMLImageElement);
            }
        },
        [onLoad, setLoadSuccess],
    );

    const handleOnErrorLoad = useCallback((): void => {
        setLoadError();

        logError(
            {
                message: `[YATRAVEL][TRAVEL_IMAGE] Ошибка загрузки изображения`,
                block: 'travelImage',
            },
            `Ошибка загрузки изображения: ${src}`,
        );

        if (onError) {
            onError(src);
        }
    }, [onError, setLoadError, src]);

    const {
        ref: responsiveRef,
        srcSet,
        sizes,
        isReady: isSrcSetReady,
    } = useSrcSet<HTMLDivElement>({srcSet: responsiveSet});

    const [inViewRef, hasIntersected] = useInView({
        ...INTERSECTION_OBSERVER_OPTIONS,
        skip: !isLoadOnViewportIntersect,
    });

    const setRefs = useCallback(
        (node: HTMLDivElement | null) => {
            inViewRef(node);

            if (!responsiveRef) {
                return;
            }

            responsiveRef.current = node;
        },
        [inViewRef, responsiveRef],
    );

    const initializeImage = useCallback(() => {
        if (!src) {
            return;
        }

        const imageInstance = new Image();

        imageInstance.src = src;

        if (sizes) {
            imageInstance.sizes = sizes;
        }

        if (srcSet) {
            imageInstance.srcset = srcSet;
        }

        imageInstance.onload = handleOnSuccessLoad;
        imageInstance.onerror = handleOnErrorLoad;
    }, [src, handleOnSuccessLoad, handleOnErrorLoad, sizes, srcSet]);

    useEffect(() => {
        if (!isSrcSetReady) {
            return;
        }

        if (isLoadOnViewportIntersect) {
            if (!hasIntersected) {
                return;
            }

            initializeImage();

            return;
        }

        if (!isPlaceholder) {
            initializeImage();
        }
    }, [
        hasIntersected,
        initializeImage,
        isLoadOnViewportIntersect,
        isPlaceholder,
        isSrcSetReady,
        responsiveSet,
    ]);

    const hasWithoutImageMod = isLoading || isError || isPlaceholder;
    const hasImageSrc = Boolean(src);

    return (
        <section
            className={cx(
                'imageContainer',
                className,
                hasWithoutImageMod && withoutImageClassName,
                {
                    imageContainer_withoutImage: hasWithoutImageMod,
                    imageContainer_fitCover: hasFitCover,
                },
            )}
            style={style}
            ref={setRefs}
            onClick={handleClick}
            itemScope={isRenderImmediately}
            itemType={
                isRenderImmediately
                    ? 'http://schema.org/ImageObject'
                    : undefined
            }
        >
            {hasImageSrc &&
                !isRenderImmediately &&
                (isLoading || isPlaceholder) && (
                    <div className={cx('loader', loaderClassName)}>
                        <Spinner size={spinnerSize} />
                    </div>
                )}
            {(!hasImageSrc || isError) && (
                <div className={cx('error', errorClassName)}>
                    {imageStub || defaultImageStub}
                </div>
            )}
            {hasImageSrc && (isRenderImmediately || isSuccess) && (
                <img
                    className={cx('image', imageClassName, {
                        image_crop: isCrop,
                        image_wide: isWide,
                    })}
                    src={src}
                    srcSet={srcSet}
                    sizes={sizes}
                    ref={imageRef}
                    alt={imageAlt}
                    style={imageStyle}
                    itemProp={isRenderImmediately ? 'contentUrl' : undefined}
                    {...prepareQaAttributes(props)}
                    {...(attrs || {})}
                />
            )}
            {Boolean(hasImageSrc && isRenderImmediately && imageAlt) && (
                <meta itemProp="name" content={imageAlt} />
            )}
        </section>
    );
};

export default TravelImage;
