import {useMemo, MutableRefObject} from 'react';

import {TImage} from 'types/common/TImage';

import {useElementSize} from 'utilities/hooks/useElementSize';
import getSrcSetString from 'utilities/images/getSrcSetString';
import getSizesAttr from '../utilities/getSizesAttr';

type TUseSrcSetResult<R extends HTMLElement> = {
    ref: MutableRefObject<R | null> | null;
    srcSet?: string;
    sizes?: string;
    isReady: boolean;
};

interface IUseSrcSetParams {
    srcSet?: TImage[];
}

/**
 * Формирует значения для атрибутов srcSet и sizes изображения, чтобы дать
 * браузерам возможность автоматически загружать нужное разрешение картинки
 * в зависимости от размера блока/разрешения экрана и т.д.
 *
 * На вход принимает srcSet – набор всех доступных размеров изображений с их размерами
 * и формирует из них строку.
 *
 * Автоматически определяет отображаемый размер контейнера, в котором будет картинка
 * и возвращает это значение, чтобы браузер мог вычислить необходимый размер для загрузки.
 *
 * Чтобы дождаться, что все замеры произошли дополнительно возвращается флаг isMeasured.
 */
export default function useSrcSet<T extends HTMLElement>({
    srcSet,
}: IUseSrcSetParams): TUseSrcSetResult<T> {
    const {ref, size, isMeasured} = useElementSize<null, T>({});

    const srcSetString = useMemo(() => getSrcSetString(srcSet), [srcSet]);

    const sizesString = useMemo(() => {
        if (!srcSet) {
            return;
        }

        const firstImage = srcSet[0];

        if (!firstImage || !firstImage.size) {
            return;
        }

        return getSizesAttr(size, firstImage.size);
    }, [srcSet, size]);

    const isReady = isMeasured || !srcSet;

    return {
        ref,
        srcSet: srcSetString,
        sizes: sizesString,
        isReady,
    };
}
