import React, {ReactNode, useCallback, useEffect} from 'react';
import ReactTruncateMarkup from 'react-truncate-markup';

import {IWithClassName} from 'types/withClassName';

import {CHAR_NBSP} from 'utilities/strings/charCodes';
import {useBoolean} from 'utilities/hooks/useBoolean';

import * as i18nBlock from 'i18n/common';

import LinkButton from 'components/LinkButton/LinkButton';
import {TLinkThemeType} from 'components/Link/Link';

import cx from './TruncateText.scss';

interface ITruncateTextProps extends IWithClassName {
    text: ReactNode;
    expanderText?: ReactNode;
    linesCount?: number;
    expanderClassName?: string;
    collapsable?: boolean;
    collapserText?: ReactNode;
    collapserClassName?: string;
    toggleTheme?: TLinkThemeType;
}

const DEFAULT_LINES_COUNT = 3;

const TruncateText: React.FC<ITruncateTextProps> = ({
    text,
    className,
    linesCount = DEFAULT_LINES_COUNT,
    expanderText = i18nBlock.readDashMore(),
    expanderClassName,
    collapsable,
    collapserText = i18nBlock.less(),
    collapserClassName,
    toggleTheme,
}) => {
    const {
        value: canRenderAllLine,
        setFalse: collapseText,
        toggle: toggleText,
    } = useBoolean(false);

    const handleExpand = useCallback(
        (event: React.MouseEvent) => {
            // Если текст располагается внутри элемента с которым можно взаимодействовать,
            // то разворачивание текста не должно триггерить основное действие
            event.stopPropagation();
            toggleText();
        },
        [toggleText],
    );

    useEffect(() => {
        collapseText();
    }, [text, collapseText]);

    const Component = canRenderAllLine ? 'div' : ReactTruncateMarkup;

    return (
        <Component
            lines={canRenderAllLine ? undefined : linesCount}
            ellipsis={
                <span>
                    {`...${CHAR_NBSP}`}
                    <LinkButton
                        className={expanderClassName}
                        theme={toggleTheme}
                        onClick={handleExpand}
                    >
                        {expanderText}
                    </LinkButton>
                </span>
            }
        >
            <div className={className}>
                {text}
                {collapsable && canRenderAllLine && (
                    <LinkButton
                        theme={toggleTheme}
                        className={cx('collapser', collapserClassName)}
                        onClick={handleExpand}
                    >
                        {collapserText}
                    </LinkButton>
                )}
            </div>
        </Component>
    );
};

export default TruncateText;
