import {
    FunctionComponent,
    RefObject,
    ReactNode,
    useCallback,
    useMemo,
    useRef,
    useEffect,
    useState,
} from 'react';
import {useSelector} from 'react-redux';
import {
    YandexSingleChatConfig,
    Widget,
    supportPopupUIFactory,
    presets,
    OnToggledEvent,
} from '@yandex-int/messenger.widget';

import {
    YANDEX_TRAVEL_BOT_ID,
    YANDEX_TRAVEL_SERVICE_ID,
    YANDEX_TRAVEL_WORKSPACE_ID,
} from './constants';

import {TMetrikaGoal} from 'utilities/metrika/types/goals/all';
import {TargetPosition} from '@yandex-int/messenger.widget/lib_cjs/types';

import {getUserInfo} from 'selectors/common/userInfoSelector';
import appVersionSelector from 'selectors/common/appVersion';

import {prepareMetadata, TEntryPoint} from './utilities/prepareMetadata';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {reachGoal} from 'utilities/metrika';

if (__CLIENT__) {
    require('@yandex-int/messenger.widget/lib/ui/supportpopup.css');
}

import './YandexMessenger.scss';

export interface IYandexMessengerRenderTriggerProps {
    ref: RefObject<HTMLElement>;
    handleClick(): void;
}

export interface IYandexMessengerCommonProps {
    entrypoint: TEntryPoint;
    metrikaGoal?: TMetrikaGoal;
    targetPosition?: TargetPosition;
    additionalData?: object;
    onVisibilityChange?: (isVisible: boolean) => void;
}

interface IYandexMessengerProps extends IYandexMessengerCommonProps {
    renderTrigger(params: IYandexMessengerRenderTriggerProps): ReactNode;
}

const YandexMessenger: FunctionComponent<IYandexMessengerProps> = ({
    entrypoint,
    metrikaGoal,
    targetPosition = 'bottom-right',
    renderTrigger,
    onVisibilityChange,
    additionalData,
}) => {
    const userInfo = useSelector(getUserInfo);
    const appVersion = useSelector(appVersionSelector);
    const deviceType = useDeviceType();
    const chatWidget = useRef<Widget | null>(null);
    const triggerRef = useRef<HTMLDivElement | null>(null);
    const lastToggleTimeRef = useRef<number>(0);
    const [isVisible, setIsVisible] = useState(false);

    const chatMetadata = useMemo(
        () =>
            prepareMetadata({
                userInfo,
                deviceType,
                appVersion,
                entrypoint,
                additionalData,
            }),
        [userInfo, deviceType, appVersion, entrypoint, additionalData],
    );

    const ui = useMemo(
        () =>
            __CLIENT__
                ? supportPopupUIFactory({
                      // Если не передавать popupTargetNode на таче, возникают проблемы, когда несколько инстансов на странице
                      // Поэтому на таче передаем popupTargetNode всегда
                      popupTargetNode: deviceType.isDesktop
                          ? undefined
                          : triggerRef.current ?? undefined,
                      popupTargetPosition: targetPosition,
                      autocloseable: true,
                  })
                : null,
        [deviceType.isDesktop, targetPosition],
    );

    const widget = useMemo(
        () =>
            ui
                ? new Widget(
                      new YandexSingleChatConfig(
                          presets.support(YANDEX_TRAVEL_BOT_ID),
                          {
                              serviceId: YANDEX_TRAVEL_SERVICE_ID,
                              workspaceId: YANDEX_TRAVEL_WORKSPACE_ID,
                              flags: {
                                  disableNavigation: '1',
                              },
                          },
                      ),
                  )
                      .setUI(ui)
                      .init()
                : null,
        [ui],
    );

    const handleGetEnv = useCallback(
        request => {
            try {
                request.response(chatMetadata);
            } catch (error) {
                request.reject(error);
            }
        },
        [chatMetadata],
    );

    const handleToggle = useCallback(
        (event: OnToggledEvent) => {
            const visible = !event.hidden;

            setIsVisible(visible);
            lastToggleTimeRef.current = Date.now();

            if (!visible && document.activeElement instanceof HTMLElement) {
                document.activeElement.blur();
            }

            if (typeof onVisibilityChange === 'function') {
                onVisibilityChange(visible);
            }
        },
        [onVisibilityChange],
    );

    const handleClick = useCallback(() => {
        if (Date.now() - lastToggleTimeRef.current < 300) {
            return;
        }

        if (isVisible) {
            chatWidget.current?.hide();
        } else {
            chatWidget.current?.show();
        }

        if (metrikaGoal) {
            reachGoal(metrikaGoal);
        }
    }, [chatWidget, metrikaGoal, isVisible]);

    useEffect(() => {
        if (widget) {
            widget.handlers.getEnv.addListener(handleGetEnv);
            chatWidget.current = widget;
        }

        if (ui) {
            ui.onToggled.addListener(handleToggle);
        }

        return (): void => {
            widget?.handlers.getEnv.removeListener(handleGetEnv);
            ui?.onToggled.removeListener(handleToggle);
        };
    }, [handleToggle, handleGetEnv, widget, ui]);

    useEffect(() => {
        if (ui) {
            ui.mount();
        }
    }, [ui]);
    useEffect(() => () => widget?.destroy(), [widget]);

    return <>{renderTrigger({ref: triggerRef, handleClick})}</>;
};

export default YandexMessenger;
