import {Component, ReactNode} from 'react';

import {IZoomableProps, Point} from '../../types';

import {coordinateChange, distance, restrict} from './utilities';
import {
    getXBoundsByViewport,
    getXYBounds,
} from 'components/Zoomable/components/AbstractZoomable/utilities/getXYBounds';

import cx from './AbstractZoomable.scss';

interface IZoomableState {
    translation: Point;
    initTranslation: Point;
    scale: number;
    minScale: number;
    canZoom: boolean;
    isInitialized: boolean;
}

export interface IStartPointerInfo {
    translation: Point;
    scale: number;
    start: Point;
}

export enum EDragDirectionMultiplier {
    DIRECT = -1, // для десктопа
    INVERSE = 1, // для мобилок
}

const CLICK_EPSILON = 10;
const INIT_TRANSLATION = {x: 0, y: 0};

export default abstract class AbstractZoomable extends Component<
    IZoomableProps,
    IZoomableState
> {
    readonly state: IZoomableState = {
        scale: 1,
        minScale: 0.1,
        canZoom: false,
        translation: INIT_TRANSLATION,
        initTranslation: INIT_TRANSLATION,
        isInitialized: false,
    };

    private viewportNode?: HTMLElement;

    protected maxScale = 1;
    protected zoomRatio = 1;
    protected zoomableNode?: HTMLElement;
    protected dragDirectionMultiplier = EDragDirectionMultiplier.INVERSE;
    private startPointerInfo?: IStartPointerInfo;

    componentDidMount(): void {
        if (window) {
            window.addEventListener('resize', this.initialize);
        }
    }

    componentDidUpdate(prevProps: IZoomableProps): void {
        if (!prevProps.childrenSize && this.props.childrenSize) {
            this.initialize();
        } else if (
            this.props.childrenSize &&
            prevProps.childrenSize &&
            (prevProps.childrenSize.width !== this.props.childrenSize.width ||
                prevProps.childrenSize.height !==
                    this.props.childrenSize.height)
        ) {
            this.initialize();
        } else if (prevProps.disableZoom !== this.props.disableZoom) {
            this.initialize();
        }
    }

    componentWillUnmount(): void {
        this.removeListeners();

        if (window) {
            window.removeEventListener('resize', this.initialize);
        }
    }

    protected abstract addListeners(): void;

    protected abstract removeListeners(): void;

    private setViewportNode = (node: HTMLDivElement): void => {
        this.viewportNode = node;

        if (this.viewportNode) {
            this.initialize();
        }
    };

    private setZoomableNode = (node: HTMLDivElement): void => {
        this.removeListeners();

        this.zoomableNode = node;

        if (!this.zoomableNode) {
            return;
        }

        this.addListeners();
    };

    private initialize = (): void => {
        this.removeListeners();

        const {childrenSize, disableZoom, onZoomChange} = this.props;
        const {viewportNode} = this;

        if (
            !viewportNode ||
            !childrenSize ||
            !childrenSize.width ||
            !childrenSize.height
        ) {
            if (this.state.isInitialized) {
                this.setState({isInitialized: false});
            }

            return;
        }

        const fitToViewportScale = Math.min(
            viewportNode.offsetWidth / childrenSize.width,
            viewportNode.offsetHeight / childrenSize.height,
        );

        const minScale = Math.min(fitToViewportScale, 1);

        this.maxScale = Math.max(1, minScale * this.zoomRatio);

        const canZoom = Boolean(!disableZoom && minScale < this.maxScale);

        if (canZoom) {
            this.addListeners();
        }

        this.setState(
            {
                minScale,
                scale: minScale,
                canZoom,
                isInitialized: false,
            },
            () => {
                const initTranslation = this.getInitTranslation(minScale);

                const translation = this.restrictTranslation(
                    initTranslation,
                    minScale,
                );

                this.setState({
                    translation,
                    initTranslation: translation,
                });
                // чтобы небыло анимации зума при первом рендере компонента
                setTimeout(
                    () =>
                        this.setState(
                            {isInitialized: true},
                            this.handleTranslationChangeEnd,
                        ),
                    100,
                );

                onZoomChange?.({
                    maxScale: this.maxScale,
                    minScale,
                    scale: minScale,
                });
            },
        );
    };

    private getInitTranslation(scale: number): Point {
        const {childrenSize} = this.props;

        if (!childrenSize || !this.viewportNode) {
            return {x: 0, y: 0};
        }

        const viewportWidth = this.viewportNode.offsetWidth;
        const viewportHeight = this.viewportNode.offsetHeight;
        const childrenWidth = childrenSize.width * scale;
        const childrenHeight = childrenSize.height * scale;

        return {
            x:
                viewportWidth > childrenWidth
                    ? (viewportWidth - childrenWidth) / 2
                    : 0,
            y:
                viewportHeight > childrenHeight
                    ? (viewportHeight - childrenHeight) / 2
                    : 0,
        };
    }

    protected beingDrag(start: Point) {
        const {scale, translation} = this.state;

        this.startPointerInfo = {
            start,
            scale,
            translation,
        };
    }

    protected dragTo(position: Point) {
        if (!this.startPointerInfo || !this.props.withDrag?.current) {
            return;
        }

        const {translation, start} = this.startPointerInfo;
        const dragX = position.x - start.x;
        const dragY = position.y - start.y;
        const newTranslation = {
            x:
                translation.x +
                dragX * this.state.scale * this.dragDirectionMultiplier,
            y:
                translation.y +
                dragY * this.state.scale * this.dragDirectionMultiplier,
        };

        const restrictedTranslation = this.restrictTranslation(
            newTranslation,
            this.state.scale,
            true,
        );

        this.setState({
            translation: restrictedTranslation,
        });
    }

    protected endDrag() {
        this.startPointerInfo = undefined;

        this.handleTranslationChangeEnd();
    }

    protected tryCallOnClick(pagePosition: Point, currentPoint?: Point) {
        if (
            this.startPointerInfo &&
            currentPoint &&
            distance(this.startPointerInfo.start, currentPoint) > CLICK_EPSILON
        ) {
            return;
        }

        const {onClick} = this.props;

        if (onClick && this.viewportNode) {
            onClick({
                currentTarget: this.viewportNode,
                pageX: pagePosition.x,
                pageY: pagePosition.y,
            });
        }
    }

    protected changeScale(scaleDelta: number, point: Point): void {
        const {minScale, scale} = this.state;
        const newScale = restrict(minScale, scale + scaleDelta, this.maxScale);

        if (scaleDelta < 0) {
            this.scaleToCenter(newScale, scaleDelta);
        } else {
            this.scaleToPoint(newScale, point);
        }
    }

    protected scaleToPoint(newScale: number, point: Point): void {
        const focus = this.getFocus(point);

        this.scaleToFocus(newScale, focus);
    }

    private getFocus(clientPos: Point): Point {
        if (!this.viewportNode) {
            return {x: 0, y: 0};
        }

        const {translation} = this.state;
        const rect = this.viewportNode.getBoundingClientRect();

        return {
            x: clientPos.x - rect.left - translation.x,
            y: clientPos.y - rect.top - translation.y,
        };
    }

    private scaleToFocus(newScale: number, focus: Point) {
        const {translation, scale} = this.state;
        const scaleRatio = newScale / (scale || 1);
        const focusPointDelta = {
            x: coordinateChange(focus.x, scaleRatio),
            y: coordinateChange(focus.y, scaleRatio),
        };
        const newTranslation = {
            x: translation.x - focusPointDelta.x,
            y: translation.y - focusPointDelta.y,
        };

        this.setState(
            {
                scale: newScale,
                translation: this.restrictTranslation(newTranslation, newScale),
            },
            this.dispatchZoomAndTranslationEvents,
        );
    }

    protected scaleToCenter(newScale: number, scaleDelta: number) {
        const {translation: oldTranslation, minScale, scale} = this.state;

        if (minScale === scale || scale === newScale) {
            return;
        }

        const initTranslation = this.getInitTranslation(minScale);

        if (newScale === minScale) {
            this.setState(
                {
                    scale: newScale,
                    translation: this.restrictTranslation(
                        initTranslation,
                        minScale,
                    ),
                },
                this.dispatchZoomAndTranslationEvents,
            );

            return;
        }

        const step = (minScale - scale) / Math.abs(scaleDelta);
        const diffX = initTranslation.x - oldTranslation.x;
        const diffY = initTranslation.y - oldTranslation.y;
        const newTranslation = {
            x: oldTranslation.x - diffX / step,
            y: oldTranslation.y - diffY / step,
        };

        this.setState(
            {
                scale: newScale,
                translation: this.restrictTranslation(newTranslation, newScale),
            },
            this.dispatchZoomAndTranslationEvents,
        );
    }

    private restrictTranslation(
        desiredTranslation: Point,
        newScale: number,
        byViewPort: boolean = false,
    ) {
        if (!this.viewportNode) {
            return desiredTranslation;
        }

        const {childrenSize} = this.props;

        if (!childrenSize || !childrenSize.width || !childrenSize.height) {
            return desiredTranslation;
        }

        const {translation} = this.state;
        const {x, y} = desiredTranslation;

        const viewportWidth = this.viewportNode.offsetWidth;
        const viewportHeight = this.viewportNode.offsetHeight;
        const {minX, maxX, minY, maxY} = getXYBounds({
            childrenSize,
            newScale,
            byViewPort,
            viewportWidth,
            translation,
            viewportHeight,
        });

        return {
            x: restrict(minX, x, maxX),
            y: restrict(minY, y, maxY),
        };
    }

    private handleTranslationChangeEnd = (): void => {
        const {onTranslationChange, childrenSize} = this.props;

        if (!onTranslationChange) {
            return;
        }

        const {scale, translation} = this.state;

        let leftBoundaryReached = false;
        let rightBoundaryReached = false;

        if (childrenSize && this.viewportNode) {
            const viewportWidth = this.viewportNode.offsetWidth;

            const {minX, maxX} = getXBoundsByViewport(
                scale,
                viewportWidth,
                childrenSize,
            );

            leftBoundaryReached = translation.x === maxX;
            rightBoundaryReached = translation.x === minX;
        }

        onTranslationChange({
            translation,
            leftBoundaryReached,
            rightBoundaryReached,
        });
    };

    private dispatchZoomChange(): void {
        const {onZoomChange} = this.props;

        if (!onZoomChange) {
            return;
        }

        const {minScale, scale} = this.state;

        onZoomChange({
            maxScale: this.maxScale,
            minScale,
            scale,
        });
    }

    private dispatchZoomAndTranslationEvents(): void {
        this.handleTranslationChangeEnd();
        this.dispatchZoomChange();
    }

    private getZoomableStyle() {
        const {childrenSize} = this.props;
        const {translation, scale} = this.state;
        const {x, y} = translation;
        const transform = `translate(${x}px, ${y}px) scale(${scale})`;

        if (!childrenSize) {
            return {transform};
        }

        return {
            transform,
            width: childrenSize.width ? `${childrenSize.width}px` : undefined,
            height: childrenSize.height
                ? `${childrenSize.height}px`
                : undefined,
        };
    }

    render(): ReactNode {
        const {children, controls} = this.props;
        const {isInitialized} = this.state;

        const zoomableStyle = this.getZoomableStyle();

        return (
            <div className={cx('root')} ref={this.setViewportNode}>
                {controls ? controls : null}
                <div
                    className={cx('zoomable', {
                        zoomable_initialized: isInitialized,
                    })}
                    style={zoomableStyle}
                    ref={this.setZoomableNode}
                >
                    {children}
                </div>
            </div>
        );
    }
}
