import React from 'react';
import {Route, RouteComponentProps, Switch, withRouter} from 'react-router-dom';
import {hot as reactHotLoaderDecorator} from 'react-hot-loader';
import {connect} from 'react-redux';
import {compose} from 'redux';

import {EProjectName} from 'constants/common';
import {ZINDEX_GROUP_LEVEL} from 'constants/layers';
import {URLs} from 'constants/urls';

import {fetchUserPlusInfo} from 'reducers/common/userInfo/thunk';
import {fetchIsSubscribed} from 'reducers/common/subscription/status/thunk';
import {DispatchProps} from 'reducers/trains/customDispatch';

import appSelector from 'selectors/appSelector';

import {params} from 'utilities/metrika';
import appData from 'utilities/appData/appData';
import {prefetchData} from 'containers/App/utilities/prefetchData';
import handleMetrikaQuery from 'utilities/metrika/handleMetrikaQuery';
import checkIntlSupport from 'containers/App/utilities/checkIntlSupport';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';

import TestAnchors from 'containers/App/components/TestAnchors/TestAnchors';
import MetrikaScript from 'containers/App/components/MetrikaScript/MetrikaScript';
import getProjectPageLoadable from 'containers/App/components/ProjectPageLoadable/ProjectPageLoadable';
import YtpRefererSaver from 'containers/App/components/YtpRefererSaver/YtpRefererSaver';
import BugScriptLoadable from 'containers/App/components/BugScript/BugScriptLoadable';
import AffiliateMarker from 'containers/AffiliateMarker/AffiliateMarker';
import FontsSwapper from 'containers/App/components/FontsSwapper/FontsSwapper';
import NetworkError from 'containers/App/components/NetworkError/NetworkError';

import BaseTravelHelmet from 'components/BaseTravelHelmet/BaseTravelHelmet';
import UxfeedbackLoadable from './components/Uxfeedback/UxfeedbackLoadable';
import PageErrorBoundary from './components/PageErrorBoundary/PageErrorBoundary';
import BottomSheetManager from 'components/BottomSheet/components/BottomSheetManager/BottomSheetManager';
import OpenRequestManagerButton from 'projects/testControlPanel/pages/TestRequestManagerPage/components/OpenRequestManagerButton/OpenRequestManagerButton';

import LayersContext from 'contexts/LayersContext';
import FeaturesContext from 'contexts/FeaturesContext';
import CurrenciesContext from 'contexts/CurrenciesContext';
import DeviceTypeContext from 'contexts/DeviceTypeContext';
import EnvironmentConfigContext from 'contexts/EnvironmentConfigContext';
import {PriceComparatorProvider} from 'contexts/PriceComparatorContext';
import {RumUiProvider} from 'contexts/RumUiContext';
import {PlatformContext} from 'contexts/PlatformContext';
import {serverFetchDataDispatcher} from 'contexts/ServerFetchDataContext';
import WhiteLabelConfigContext from 'contexts/WhiteLabelConfigContext';

import cx from './App.scss';

type TAppProps = ReturnType<typeof appSelector> &
    DispatchProps &
    RouteComponentProps;

class App extends React.Component<TAppProps> {
    componentDidMount(): void {
        const {
            dispatch,
            location,
            features,
            userIsAuth,
            userHasPlusInfo,
            userHasPlus,
            subscribeStatusFetched,
        } = this.props;
        const query = getQueryByLocation(location);

        this.trySendHero();
        handleMetrikaQuery(query);
        params({
            features,
            common: {
                userLoggedIn: userIsAuth,
                isPlusUser: Boolean(userHasPlus),
            },
        });

        checkIntlSupport();

        if (userIsAuth && !userHasPlusInfo) {
            dispatch(fetchUserPlusInfo());
        }

        if (!subscribeStatusFetched) {
            dispatch(fetchIsSubscribed());
        }
    }

    trySendHero(): void {
        window.Ya?.Rum?.sendHeroElement?.();
    }

    renderProject(project: EProjectName): () => React.ReactNode {
        const Component = getProjectPageLoadable(project);

        return (): React.ReactNode => (
            <PageErrorBoundary project={project}>
                <Component />
            </PageErrorBoundary>
        );
    }

    renderProjects(): React.ReactNode {
        const isWhiteLabel = Boolean(this.props.whiteLabelConfig);
        const indexAppPaths = [
            URLs.index,
            URLs.avia,
            URLs.trains,
            URLs.hotels,
            URLs.buses,
            URLs.tours,
        ];

        return (
            <Switch>
                <Route
                    exact
                    path={indexAppPaths}
                    render={this.renderProject(
                        isWhiteLabel
                            ? EProjectName.WHITE_LABEL
                            : EProjectName.INDEX,
                    )}
                />
                <Route
                    path={URLs.avia}
                    render={this.renderProject(EProjectName.AVIA)}
                />
                <Route
                    path={URLs.hotels}
                    render={this.renderProject(EProjectName.HOTELS)}
                />
                <Route
                    path={URLs.hotel}
                    render={this.renderProject(EProjectName.HOTEL)}
                />
                <Route
                    path={URLs.account}
                    render={this.renderProject(EProjectName.ACCOUNT)}
                />
                <Route
                    path={URLs.buses}
                    render={this.renderProject(EProjectName.BUSES)}
                />
                <Route
                    path={URLs.trains}
                    render={this.renderProject(EProjectName.TRAINS)}
                />
                <Route
                    path={URLs[EProjectName.SUBSCRIPTION]}
                    render={this.renderProject(EProjectName.SUBSCRIPTION)}
                />
                <Route
                    path={URLs[EProjectName.PARTNERS]}
                    render={this.renderProject(EProjectName.PARTNERS)}
                />
                <Route
                    path={URLs[EProjectName.FAVORITES]}
                    render={this.renderProject(EProjectName.FAVORITES)}
                />
                <Route
                    path={URLs[EProjectName.HOTELS_EXTRANET]}
                    render={this.renderProject(EProjectName.HOTELS_EXTRANET)}
                />
                {!appData.isProductionEnv && (
                    <Route
                        path={[
                            URLs.testControlPanelRoot,
                            URLs.testControlPanelAviaLegacyTestContext,
                        ]}
                        render={this.renderProject(
                            EProjectName.TEST_CONTROL_PANEL,
                        )}
                    />
                )}
                <Route
                    path={URLs.journalRoot}
                    render={this.renderProject(EProjectName.JOURNAL)}
                />
                <Route render={this.renderProject(EProjectName.NOT_FOUND)} />
            </Switch>
        );
    }

    render(): React.ReactNode {
        const {
            features,
            deviceType,
            currencies,
            environmentConfig,
            platform,
            needToShowBug,
            isFromXredirect,
            whiteLabelConfig,
        } = this.props;
        const {isMobile} = deviceType;

        return (
            <EnvironmentConfigContext.Provider value={environmentConfig}>
                <DeviceTypeContext.Provider value={deviceType}>
                    <FeaturesContext.Provider value={features}>
                        <CurrenciesContext.Provider value={currencies}>
                            <LayersContext.Provider value={ZINDEX_GROUP_LEVEL}>
                                <WhiteLabelConfigContext.Provider
                                    value={whiteLabelConfig}
                                >
                                    <BottomSheetManager>
                                        <PriceComparatorProvider>
                                            <PlatformContext.Provider
                                                value={platform}
                                            >
                                                {!isFromXredirect && (
                                                    <BaseTravelHelmet />
                                                )}
                                                {!__DEV__ && <MetrikaScript />}
                                                {needToShowBug && (
                                                    <BugScriptLoadable />
                                                )}
                                                <FontsSwapper />
                                                <div
                                                    className={cx('App', {
                                                        App_mobile: isMobile,
                                                    })}
                                                >
                                                    <TestAnchors />
                                                    <RumUiProvider>
                                                        {this.renderProjects()}
                                                    </RumUiProvider>
                                                    {!appData.isProductionEnv && (
                                                        <OpenRequestManagerButton />
                                                    )}
                                                </div>
                                                <YtpRefererSaver />
                                                <AffiliateMarker />
                                                <UxfeedbackLoadable />
                                                <NetworkError />
                                            </PlatformContext.Provider>
                                        </PriceComparatorProvider>
                                    </BottomSheetManager>
                                </WhiteLabelConfigContext.Provider>
                            </LayersContext.Provider>
                        </CurrenciesContext.Provider>
                    </FeaturesContext.Provider>
                </DeviceTypeContext.Provider>
            </EnvironmentConfigContext.Provider>
        );
    }
}

export default compose(
    reactHotLoaderDecorator(module),
    withRouter,
    connect(appSelector),
    serverFetchDataDispatcher([prefetchData], {
        checkNested: true,
    }),
)(App);
